/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.model;

import static org.assertj.core.api.Assertions.assertThat;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class UserTest {

	private User user;
	private User other;

	@BeforeEach
	public void init() {
		user = new User("user1", "", "user1", "email", DefaultRole.ROLE_STUDENT, 1);
		other = new User("user1", "", "user1", "email", DefaultRole.ROLE_STUDENT, 1);
		user.id = 1L;
		other.id = 1L;
	}

	@Test
	public void equalsSameUser() {
		assertThat(user.equals(user)).isTrue();
	}

	@Test
	public void equalsNull() {
		assertThat(user.equals(null)).isFalse();
	}

	@Test
	public void equalsOtherInstance() {
		assertThat(user.equals(new Course())).isFalse();
	}

	@Test
	public void equalsUserIdNull() {
		user.id = null;
		assertThat(user.equals(other)).isFalse();
	}

	@Test
	public void equalsBothIdNull() {
		user.id = null;
		other.id = null;
		assertThat(user.equals(other)).isTrue();
	}

	@Test
	public void equalsOtherId() {
		user.id = 2L;
		assertThat(user.equals(other)).isFalse();
	}

	@Test
	public void equalsUsernameNull() {
		user.setUsername(null);
		assertThat(user.equals(other)).isFalse();
	}

	@Test
	public void equalsBothUsernameNull() {
		user.setUsername(null);
		other.setUsername(null);
		assertThat(user.equals(other)).isTrue();
	}

	@Test
	public void equalsOtherUsername() {
		user.setUsername("user2");
		assertThat(user.equals(other)).isFalse();
	}

	@Test
	public void equalsSimple() {
		assertThat(user.equals(other)).isTrue();
	}

	@Test
	public void hashCodeSimple() {
		assertThat(user.hashCode()).isEqualTo(111579558);
	}

	@Test
	public void hashCodeIdNull() {
		user.id = null;
		assertThat(user.hashCode()).isEqualTo(111579527);
	}

	@Test
	public void hashCodeUsernameNull() {
		user.setUsername(null);
		assertThat(user.hashCode()).isEqualTo(992);
	}

	@Test
	public void addRoleRightUser() {
		Role role = new Assistant(user, new Course());
		user.addRole(role);

		assertThat(user.getRoles().contains(role)).isTrue();
	}

	@Test
	public void addRoleWrongUser() {
		Role role = new Assistant(other, new Course());
		user.id = 2L;
		user.addRole(role);

		assertThat(user.getRoles().contains(role)).isTrue();
		assertThat(role.getUser().equals(other)).isFalse();
	}

	@Test
	public void isAdminFalse() {
		assertThat(user.isAdmin()).isFalse();
	}

	@Test
	public void isAdminTrue() {
		user.setDefaultRole(DefaultRole.ROLE_ADMIN);
		assertThat(user.isAdmin()).isTrue();
	}

	@Test
	public void isTeacherFalse() {
		assertThat(user.isTeacher()).isFalse();
	}

	@Test
	public void isTeacherTrue() {
		user.setDefaultRole(DefaultRole.ROLE_TEACHER);
		assertThat(user.isTeacher()).isTrue();
	}

	@Test
	public void isStudentFalse() {
		user.setDefaultRole(DefaultRole.ROLE_ADMIN);
		assertThat(user.isStudent()).isFalse();
	}

	@Test
	public void isStudentTrue() {
		assertThat(user.isStudent()).isTrue();
	}

}
