/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.transaction.Transactional;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.exception.AlreadyEnqueuedException;
import nl.tudelft.ewi.queue.exception.InvalidSlotException;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
@Transactional
public class LabServiceTest {

	@Autowired
	LabRepository labRepository;

	@Autowired
	LabService labService;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private RoomRepository roomRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private AssignmentRepository assignmentRepository;

	@Autowired
	private FirstYearMentorGroupRepository firstYearMentorGroupRepository;

	private Lab lab;

	@BeforeEach
	public void init() {
		Lab firstLab = labRepository.findAll().get(0);
		lab = new Lab();
		lab.setAssignments(firstLab.getAssignments());
		lab.setRooms(firstLab.getRooms());
		lab.setCommunicationMethod(CommunicationMethod.TA_VISIT_STUDENT);
		lab.setAllowedRequestTypes(firstLab.getAllowedRequestTypes());
	}

	/**
	 * Test to make sure the repeatLabFor method works without throwing any errors.
	 */
	@Test
	public void repeatLabForTest() {
		Lab temp = labRepository.findAll().get(0);
		int initialSize = labRepository.findAll().size();

		labService.repeatLabFor(2, temp);

		assertThat(labRepository.findAll().size()).isEqualTo(initialSize + 2);
	}

	@Test
	public void repeatLabForSignOffIntervals() {
		Lab temp = labRepository.findAll().get(0);
		temp.setSlotSelectionOpensAt(LocalDateTime.now());
		temp.setIntervalTime(15L);
		temp.setCapacity(1L);
		temp.setSignOffIntervals(true);
		int initialSize = labRepository.findAll().size();

		labService.repeatLabFor(1, temp);

		assertThat(labRepository.findAll().size()).isEqualTo(initialSize + 1);
	}

	@Test
	public void testLabSaving() {
		lab.setTitle("test");
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		LocalDateTime startOfLab = LocalDateTime.now().plusMinutes(1);
		lab.setExamLab(true);
		lab.setExamLabPercentage(10);
		LabSlot labSlot = new LabSlot(startOfLab, startOfLab.plusHours(4));
		lab.setSlot(labSlot);
		lab.setIntervalTime(10L);
		lab.setSignOffIntervals(true);
		lab.setCapacity(2L);
		Course course = courseRepository.findByIdOrThrow(1L);
		course.setLabs(Collections.singletonList(lab));
		Room room = roomRepository.findAll().get(0);
		Assignment assignment = assignmentRepository.findAll().iterator().next();
		lab.setCourse(course);
		lab.setRooms(Collections.singletonList(room));
		lab.setAssignments(Collections.singletonList(assignment));
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAllowedRequestTypes(Collections.singletonList(new RequestType("rt")));

		int numberOfLabs = labRepository.findAll().size();

		labService.saveLab(course, lab, Optional.empty());

		Lab newLab = labRepository.findByIdOrThrow(lab.getId());

		assertThat(labRepository.findAll().size()).isEqualTo(numberOfLabs + 1);
		assertThat(newLab).isNotNull();
		assertThat(newLab.getTitle()).isEqualTo("test");

	}

	@Test
	public void enqueueSimple() {
		Course course = courseRepository.findAll().get(0);
		User user = new User();

		lab.setCourse(course);
		lab.setSlot(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now().plusHours(1)));

		labRepository.save(lab);

		user.addRole(new Student(user, course, LocalDateTime.now()));

		Request request = new Request();
		request.setRequestEntity(user);
		request.setLab(lab);
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));
		request.setAssignment(lab.getAssignments().get(0));

		labService.enqueue(request);

		Optional<Request> r = requestRepository.findById(request.getId());
		assertThat(r.isPresent()).isTrue();

		labRepository.delete(lab);
	}

	@Test
	public void enqueueNoSignOffIntervalsNotOpen() {
		lab.setSignOffIntervals(false);
		lab.setSlot(new LabSlot(LocalDateTime.now().plusHours(1), LocalDateTime.now().plusDays(1)));

		Request request = new Request();
		request.setLab(lab);

		assertThrows(AccessDeniedException.class, () -> labService.enqueue(request));
	}

	@Test
	public void enqueueUserNotParticipatesCourse() {
		lab.setCourse(new Course());
		lab.setSignOffIntervals(true);

		Request request = new Request();
		request.setRequestEntity(new User());
		request.setLab(lab);

		assertThrows(AccessDeniedException.class, () -> labService.enqueue(request));
	}

	@Test
	public void enqueueUserAlreadyEnqueued() {
		Course course = new Course();
		User user = new User();

		lab.setCourse(course);
		lab.setSignOffIntervals(true);

		user.addRole(new Student(user, course, LocalDateTime.now()));

		Request request = new Request();
		request.setRequestEntity(user);
		request.setLab(lab);
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));
		request.setAssignment(lab.getAssignments().get(0));

		labService.enqueue(request);
		assertThrows(AlreadyEnqueuedException.class, () -> labService.enqueue(request));
	}

	@Test
	public void enqueueFeedbackLabSimple() {
		Course course = courseRepository.findAll().get(0);
		User user = new User();

		lab.setCourse(course);
		lab.setSlot(new LabSlot(LocalDateTime.now().minusHours(1), LocalDateTime.now().plusHours(1)));
		lab.setFeedbackLab(true);

		labRepository.save(lab);

		user.addRole(new Student(user, course, LocalDateTime.now()));

		Request request = new Request();
		request.setRequestEntity(user);
		request.setLab(lab);
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));
		request.setAssignment(lab.getAssignments().get(0));

		labService.enqueue(request);

		Optional<Request> r = requestRepository.findById(request.getId());
		assertThat(r.isPresent()).isTrue();

		labRepository.delete(lab);
	}

	@Test
	public void enqueueSlotBiggerThanInterval() {
		Course course = new Course();
		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));

		lab.setIntervalTime(10L);
		lab.setFeedbackLab(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1)));
		lab.setCourse(course);

		Request request = new Request();
		request.setSlot(new RequestSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1)));
		request.setLab(lab);
		request.setRequestEntity(user);

		assertThrows(InvalidSlotException.class, () -> labService.enqueue(request));
	}

	@Test
	public void enqueueSlotStartsBeforeLab() {
		Course course = new Course();
		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));

		lab.setIntervalTime(100L);
		lab.setFeedbackLab(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1)));
		lab.setCourse(course);

		Request request = new Request();
		request.setSlot(
				new RequestSlot(LocalDateTime.now().minusMinutes(5), LocalDateTime.now().plusMinutes(5)));
		request.setLab(lab);
		request.setRequestEntity(user);

		assertThrows(InvalidSlotException.class, () -> labService.enqueue(request));
	}

	@Test
	public void enqueueSlotNotAvailable() {
		Course course = new Course();
		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));

		lab.setIntervalTime(100L);
		lab.setFeedbackLab(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1)));
		lab.setCourse(course);
		lab.setCapacity(0L);

		Request request = new Request();
		request.setSlot(
				new RequestSlot(LocalDateTime.now().plusMinutes(5), LocalDateTime.now().plusMinutes(15)));
		request.setLab(lab);
		request.setRequestEntity(user);

		assertThrows(InvalidSlotException.class, () -> labService.enqueue(request));
	}

	@Test
	public void enqueueCorrectSlot() {
		Course course = courseRepository.findAll().get(0);
		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));

		lab.setIntervalTime(100L);
		lab.setFeedbackLab(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1)));
		lab.setCourse(course);
		lab.setCapacity(100L);

		labRepository.save(lab);

		Request request = new Request();
		request.setSlot(
				new RequestSlot(LocalDateTime.now().plusMinutes(5), LocalDateTime.now().plusMinutes(15)));
		request.setLab(lab);
		request.setRequestEntity(user);
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));
		request.setAssignment(lab.getAssignments().get(0));

		labService.enqueue(request);

		Optional<Request> r = requestRepository.findById(request.getId());
		assertThat(r.isPresent()).isTrue();

		labRepository.delete(lab);
	}

	@Test
	public void saveLabToBeDeterminedRoomNotExists() {
		Course course = new Course();
		lab.setSignOffIntervals(true);

		Room room = roomRepository.findByName("To be determined");
		if (room != null)
			roomRepository.delete(room);

		labService.saveLab(course, lab, Optional.empty());

		Optional<Lab> l = labRepository.findById(lab.getId());
		assertThat(l.isPresent()).isTrue();
	}

	@Test
	public void saveLabToBeDeterminedRoomExists() {
		Course course = new Course();
		lab.setSignOffIntervals(true);

		Room room = roomRepository.findByName("To be determined");
		if (room == null)
			roomRepository.save(new Room("To be determined"));

		labService.saveLab(course, lab, Optional.empty());

		Optional<Lab> l = labRepository.findById(lab.getId());
		assertThat(l.isPresent()).isTrue();
	}

	@Test
	public void saveLabAssignmentNull() {
		Course course = new Course();
		List<Assignment> assignments = new ArrayList<>();
		assignments.add(null);
		lab.setAssignments(assignments);

		labService.saveLab(course, lab, Optional.empty());

		Optional<Lab> l = labRepository.findById(lab.getId());
		assertThat(l.isPresent()).isTrue();
	}

	@Test
	public void copyMentorGroups() {
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(1)));
		FirstYearMentorGroup fymg = new FirstYearMentorGroup("fymg");
		firstYearMentorGroupRepository.save(fymg);

		lab.setCourse(courseRepository.findAll().get(0));
		lab.setAllowedMentorGroups(Collections.singletonList(fymg));

		int size = labRepository.findAll().size();

		labService.repeatLabFor(1, lab);
		assertThat(labRepository.findAll().size()).isEqualTo(size + 1);
	}

	@Test
	public void getIntervalsForLabNoSignOffIntervals() {
		lab.setSignOffIntervals(false);
		assertThrows(AssertionError.class, () -> labService.getIntervalsForLab(lab));
	}

	@Test
	public void getIntervalsForLabSimple() {
		lab.setSignOffIntervals(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));
		lab.setIntervalTime(15L);
		lab.setCapacity(1L);

		ArrayList<RequestSlot> requestSlots = labService.getIntervalsForLab(lab);
		assertThat(requestSlots.size()).isEqualTo(16);
	}

	@Test
	public void randomEnqueueNoLabs() {
		Optional<Lab> l = labService.randomEnqueue(new Assignment(), new User());
		assertThat(l.isEmpty()).isTrue();
	}

	@Test
	public void randomEnqueueUser() {
		Course course = courseRepository.findAll().get(0);
		lab.setCourse(course);
		Assignment assignment = new Assignment(course, "test");
		assignmentRepository.save(assignment);
		lab.setSignOffIntervals(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));
		lab.setAllowWithoutMentorGroup(true);
		lab.setIntervalTime(15L);
		lab.setCapacity(1L);
		lab.setAssignments(Collections.singletonList(assignment));
		lab.setAllowedRequestTypes(Collections.singletonList(new RequestType("test")));
		Room room = new Room("test");
		roomRepository.save(room);
		lab.setRooms(Collections.singletonList(room));
		labRepository.save(lab);

		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));

		assertThrows(NullPointerException.class, () -> labService.randomEnqueue(assignment, user));
	}

	@Test
	public void randomEnqueueGroup() {
		Course course = courseRepository.findAll().get(0);
		lab.setCourse(course);
		Assignment assignment = new Assignment(course, "test");
		assignmentRepository.save(assignment);
		lab.setSignOffIntervals(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));
		lab.setAllowWithoutMentorGroup(true);
		lab.setIntervalTime(15L);
		lab.setCapacity(1L);
		lab.setAssignments(Collections.singletonList(assignment));
		lab.setAllowedRequestTypes(Collections.singletonList(new RequestType("test")));

		Room room = new Room("test");
		roomRepository.save(room);
		lab.setRooms(Collections.singletonList(room));
		labRepository.save(lab);

		Group group = new Group();
		group.setCourse(course);
		labService.randomEnqueue(assignment, group);
	}

	@Test
	public void randomEnqueueRequestExistsSameAssignment() {
		Course course = courseRepository.findAll().get(0);
		lab.setCourse(course);
		Assignment assignment = new Assignment(course, "test");
		assignmentRepository.save(assignment);
		lab.setSignOffIntervals(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));
		lab.setAllowWithoutMentorGroup(true);
		lab.setIntervalTime(15L);
		lab.setCapacity(1L);
		lab.setAssignments(Collections.singletonList(assignment));
		lab.setAllowedRequestTypes(Collections.singletonList(new RequestType("test")));

		Room room = new Room("test");
		roomRepository.save(room);
		lab.setRooms(Collections.singletonList(room));

		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));
		userRepository.save(user);

		Request request = new Request();
		Lab tempLab = labRepository.findAll().get(0);

		request.setAssignment(assignment);
		request.setRequestEntity(user);
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(tempLab.getAllowedRequestTypes().get(0));
		request.setStatus(Request.Status.PENDING);
		lab.addRequest(request);

		labRepository.save(lab);

		assertThrows(NullPointerException.class, () -> labService.randomEnqueue(assignment, user));
	}

	@Test
	public void randomEnqueueRequestExistsWrongAssignment() {
		Course course = courseRepository.findAll().get(0);
		lab.setCourse(course);
		Assignment assignment = new Assignment(course, "test");
		assignmentRepository.save(assignment);
		lab.setSignOffIntervals(true);
		lab.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));
		lab.setAllowWithoutMentorGroup(true);
		lab.setIntervalTime(15L);
		lab.setCapacity(1L);
		lab.setAssignments(Collections.singletonList(assignment));
		lab.setAllowedRequestTypes(Collections.singletonList(new RequestType("test")));

		Room room = new Room("test");
		roomRepository.save(room);
		lab.setRooms(Collections.singletonList(room));

		User user = new User();
		user.addRole(new Student(user, course, LocalDateTime.now()));
		userRepository.save(user);

		Request request = new Request();
		Assignment assignment1 = new Assignment(course, "test1");
		assignmentRepository.save(assignment1);
		Lab tempLab = labRepository.findAll().get(0);

		request.setAssignment(assignment1);
		request.setRequestEntity(user);
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(tempLab.getAllowedRequestTypes().get(0));
		request.setStatus(Request.Status.PENDING);
		lab.addRequest(request);

		labRepository.save(lab);

		assertThrows(AlreadyEnqueuedException.class, () -> labService.randomEnqueue(assignment, user));
	}
}
