/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

import java.time.LocalDateTime;
import java.util.List;
import java.util.TreeSet;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class LabStatusServiceTest {

	@Autowired
	private LabStatusService labStatusService;

	private TreeSet<Long> buckets;

	@BeforeEach
	public void init() {
		Lab lab1 = new Lab();
		lab1.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));

		Lab lab2 = new Lab();
		lab2.setSlot(new LabSlot(LocalDateTime.now().plusHours(1), LocalDateTime.now().plusHours(5)));
		buckets = labStatusService.createBucketsOverCourse(List.of(lab1, lab2), 2);
	}

	@Test
	public void countHandledRequestsInBucketsSimple() {
		Request handled = new Request();
		handled.setStatus(Request.Status.APPROVED);
		handled.setApprovedAt(LocalDateTime.of(2020, 1, 1, 0, 0));

		Request handledNoTime = new Request();
		handledNoTime.setStatus(Request.Status.APPROVED);

		Request nonHandled = new Request();
		nonHandled.setStatus(Request.Status.PENDING);

		List<Request> requests = List.of(handled, handledNoTime, nonHandled);

		List<Long> vals = labStatusService.countHandledRequestsInBuckets(buckets, requests);
		assertThat(vals.get(0)).isEqualTo(1L);
		assertThat(vals.get(1)).isEqualTo(0L);
	}

	@Test
	public void countOpenRequestsInBucketsSimple() {
		Request notArchived = new Request();
		notArchived.setCreatedAt(LocalDateTime.now());
		notArchived.setStatus(Request.Status.PENDING);

		Request earlyCreationTime = new Request();
		earlyCreationTime.setStatus(Request.Status.PENDING);
		earlyCreationTime.setCreatedAt(LocalDateTime.now().minusMinutes(1));

		Request lateCreationTime = new Request();
		lateCreationTime.setStatus(Request.Status.PENDING);
		lateCreationTime.setCreatedAt(LocalDateTime.now().minusMinutes(1));

		Request noArchivedTime = new Request();
		noArchivedTime.setStatus(Request.Status.APPROVED);
		noArchivedTime.setCreatedAt(LocalDateTime.now());

		Request archived = new Request();
		archived.setStatus(Request.Status.APPROVED);
		archived.setCreatedAt(LocalDateTime.now());
		archived.setApprovedAt(LocalDateTime.now().minusMinutes(1));

		List<Request> requests = List.of(notArchived, earlyCreationTime, lateCreationTime, noArchivedTime,
				archived);

		labStatusService.countOpenRequestsInBuckets(buckets, requests);
	}

	@Test
	public void countDistinctAssistants() {
		User assistant1 = Mockito.mock(User.class);
		when(assistant1.getId()).thenReturn(1L);

		User assistant2 = Mockito.mock(User.class);
		when(assistant2.getId()).thenReturn(2L);

		Request request1 = new Request();
		request1.setAssistant(assistant1);
		Request request2 = new Request();
		request2.setAssistant(assistant1);
		Request request3 = new Request();
		request3.setAssistant(assistant2);
		Request request4 = new Request();
		request4.setAssistant(null);

		List<Request> requests = List.of(request1, request2, request3, request4);

		assertThat(labStatusService.countDistinctAssistants(requests)).isEqualTo(2L);
	}

	@Test
	public void averageWaitingTimeSimple() {
		Request notProcessingOrHandled = new Request();
		notProcessingOrHandled.setStatus(Request.Status.PENDING);

		Request processing = new Request();
		processing.setStatus(Request.Status.PROCESSING);

		Request handled = new Request();
		handled.setStatus(Request.Status.APPROVED);

		Request handledWithProcessingTimeNoSlot = new Request();
		handledWithProcessingTimeNoSlot.setStatus(Request.Status.APPROVED);
		handledWithProcessingTimeNoSlot.setProcessingAt(LocalDateTime.now());
		handledWithProcessingTimeNoSlot.setCreatedAt(LocalDateTime.now());

		Request handledWithProcessingTimeWithSlot = new Request();
		handledWithProcessingTimeWithSlot.setStatus(Request.Status.APPROVED);
		handledWithProcessingTimeWithSlot
				.setSlot(new RequestSlot(LocalDateTime.of(2020, 1, 1, 0, 0), LocalDateTime.now()));
		handledWithProcessingTimeWithSlot.setProcessingAt(LocalDateTime.of(2020, 1, 1, 1, 0));

		List<Request> requests = List.of(notProcessingOrHandled, processing, handled,
				handledWithProcessingTimeNoSlot, handledWithProcessingTimeWithSlot);

		assertThat(labStatusService.averageWaitingTime(requests)).isEqualTo(1800L);
	}

	@Test
	public void averageProcessingTimeSimple() {
		Request notHandled = new Request();
		notHandled.setStatus(Request.Status.PENDING);

		Request noProcessingTime = new Request();
		noProcessingTime.setStatus(Request.Status.APPROVED);

		Request noHandledTime = new Request();
		noHandledTime.setStatus(Request.Status.APPROVED);
		noHandledTime.setProcessingAt(LocalDateTime.now());

		Request processed = new Request();
		processed.setStatus(Request.Status.APPROVED);
		processed.setProcessingAt(LocalDateTime.now());
		processed.setApprovedAt(LocalDateTime.now().plusSeconds(60));

		List<Request> requests = List.of(notHandled, noProcessingTime, noHandledTime, processed);

		assertThat(labStatusService.averageProcessingTime(requests)).isEqualTo(60L);
	}
}
