/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import static org.assertj.core.api.Java6Assertions.assertThat;
import static org.mockito.Mockito.when;

import java.util.Arrays;
import java.util.Map;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.transaction.annotation.Transactional;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
public class NotificationServiceTest {

	@Autowired
	private NotificationService notificationService;

	private Lab lab;
	private User assistant;
	private User student;
	private Request request;
	private Group group;

	@BeforeEach
	public void before() {
		lab = Mockito.mock(Lab.class);
		assistant = Mockito.mock(User.class);
		student = Mockito.mock(User.class);
		request = Mockito.mock(Request.class);
		group = Mockito.mock(Group.class);
	}

	/**
	 * Make sure that the correct notification is created for a feedback notification. This includes the
	 * correct url, message and the correct number of notifications for 1 user.
	 */
	@Test
	@Transactional
	public void createFeedbackNotificationTest() {
		when(request.getId()).thenReturn(1L);
		when(request.getLab()).thenReturn(lab);
		when(lab.getCourse()).thenReturn(new Course());

		Map<User, Notification> notificationMap = notificationService.createFeedbackNotification(student,
				request);

		assertThat(notificationMap.size()).isEqualTo(1);
		assertThat(notificationMap.keySet()).containsExactly(student);
		assertThat(notificationMap.get(student).getUrl()).isEqualTo("/request/1");
		assertThat(notificationMap.get(student).getMessage())
				.isEqualTo("Please provide feedback for your TA.");
	}

	/**
	 * Make sure the correct number of notifications are created for a group and that the correct notification
	 * is assigned to the correct student.
	 */
	@Test
	@Transactional
	public void createFeedbackNotificationForGroupsTest() {
		when(request.getId()).thenReturn(1L);
		when(request.getLab()).thenReturn(lab);
		when(lab.getCourse()).thenReturn(new Course());
		when(group.isGroup()).thenReturn(true);
		when(group.getMembers()).thenReturn(Arrays.asList(student, assistant));

		Map<User, Notification> notificationMap = notificationService.createFeedbackNotification(group,
				request);

		assertThat(notificationMap.size()).isEqualTo(2);
		assertThat(notificationMap.keySet()).containsExactlyInAnyOrder(student, assistant);
		assertThat(notificationMap.get(student).getUser()).isEqualTo(student);
		assertThat(notificationMap.get(assistant).getUser()).isEqualTo(assistant);
	}

	@Test
	@Transactional
	public void createNotFoundNotification() {
		when(request.getId()).thenReturn(1L);
		when(group.isGroup()).thenReturn(false);
		when(request.getLab()).thenReturn(lab);
		when(lab.getCourse()).thenReturn(new Course());

		Map<User, Notification> notificationMap = notificationService.createNotFoundNotification(student,
				request);

		assertThat(notificationMap.size()).isEqualTo(1);
		assertThat(notificationMap.keySet()).containsExactly(student);
		assertThat(notificationMap.get(student).getMessage()).isEqualTo(
				"The TA was not able to find you. Please check if you filled in the correct room! Remember to respond to the TA when he/she calls your name!");
	}

	@Test
	@Transactional
	public void createNotFoundNotificationsGroup() {
		when(request.getId()).thenReturn(1L);
		when(group.isGroup()).thenReturn(true);
		when(request.getLab()).thenReturn(lab);
		when(lab.getCourse()).thenReturn(new Course());
		when(group.getMembers()).thenReturn(Arrays.asList(student, assistant));

		Map<User, Notification> notificationMap = notificationService.createNotFoundNotification(group,
				request);

		assertThat(notificationMap.size()).isEqualTo(2);
		assertThat(notificationMap.keySet()).containsExactlyInAnyOrder(student, assistant);
		assertThat(notificationMap.get(student).getUser()).isEqualTo(student);
		assertThat(notificationMap.get(assistant).getUser()).isEqualTo(assistant);
	}

}
