/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.time.LocalDateTime;
import java.util.*;

import javax.persistence.EntityNotFoundException;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.transaction.annotation.Transactional;

@SpringBootTest(classes = QueueApplication.class)
public class PermissionServiceTest {

	@Autowired
	private PermissionService permissionService;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private NotificationRepository notificationRepository;

	@Autowired
	private RequestRepository requestRepository;

	private Object principal;

	@BeforeEach
	public void setup() {
		principal = SecurityContextHolder.getContext().getAuthentication().getPrincipal();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	public void isAdminAdminTest() {
		assertThat(permissionService.isAdmin(principal)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	public void isAdminTeacherTest() {
		assertThat(permissionService.isAdmin(principal)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewCourseTrue() {
		assertThat(permissionService.canViewCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student100@tudelft.nl")
	@Transactional
	public void canViewCourseFalse() {
		assertThat(permissionService.canViewCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewCourseAssistant() {
		assertThat(permissionService.canViewCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewCourseManager() {
		assertThat(permissionService.canViewCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewCourseTeacher() {
		assertThat(permissionService.canViewCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewCourseAdmin() {
		assertThat(permissionService.canViewCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewArchivedCourse() {
		Course course = new Course("archived", "archived", "", false, true);
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		userPrincipal.getUser().addRole(new Student(userPrincipal.getUser(), course, LocalDateTime.now()));
		courseRepository.save(course);
		assertThat(permissionService.canViewCourse(principal, course.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canLeaveCourseTrue() {
		assertThat(permissionService.canLeaveCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student100@tudelft.nl")
	@Transactional
	public void canLeaveCourseFalse() {
		assertThat(permissionService.canLeaveCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canLeaveCourseAssistant() {
		assertThat(permissionService.canLeaveCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canLeaveCourseManager() {
		assertThat(permissionService.canLeaveCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canLeaveCourseTeacherTrue() {
		assertThat(permissionService.canLeaveCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canLeaveCourseTeacherFalse() {
		Course course = courseRepository.findByIdOrThrow(3L);
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		userPrincipal.getUser().addRole(new Teacher(userPrincipal.getUser(), course));
		assertThat(permissionService.canLeaveCourse(principal, 3L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canManageParticipantsAdmin() {
		assertThat(permissionService.canManageParticipants(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canManageParticipantsTeacher() {
		assertThat(permissionService.canManageParticipants(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canManageParticipantsManager() {
		assertThat(permissionService.canManageParticipants(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canManageParticipantsAssistant() {
		assertThat(permissionService.canManageParticipants(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canManageParticipantsStudent() {
		assertThat(permissionService.canManageParticipants(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canManageTeachersAdmin() {
		assertThat(permissionService.canManageTeachers(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canManageTeachersTeacher() {
		assertThat(permissionService.canManageTeachers(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canManageTeachersManager() {
		assertThat(permissionService.canManageTeachers(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canManageTeachersAssistant() {
		assertThat(permissionService.canManageTeachers(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canManageTeachersStudent() {
		assertThat(permissionService.canManageTeachers(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canUpdateParticipantAdminStudent() {
		assertThat(permissionService.canUpdateParticipant(principal, 1L, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canUpdateParticipantTeacherStudent() {
		assertThat(permissionService.canUpdateParticipant(principal, 1L, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canUpdateParticipantManagerStudent() {
		assertThat(permissionService.canUpdateParticipant(principal, 1L, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canUpdateParticipantManagerTeacher() {
		assertThat(permissionService.canUpdateParticipant(principal, 1L, 120L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canUpdateParticipantStudentStudent() {
		assertThat(permissionService.canUpdateParticipant(principal, 1L, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canAddParticipantsAdmin() {
		assertThat(permissionService.canAddParticipant(principal, 1L, "student")).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canAddParticipantsTeacher() {
		assertThat(permissionService.canAddParticipant(principal, 1L, "student")).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canAddParticipantsManager() {
		assertThat(permissionService.canAddParticipant(principal, 1L, "student")).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canAddParticipantsAssistant() {
		assertThat(permissionService.canAddParticipant(principal, 1L, "student")).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canAddParticipantsStudent() {
		assertThat(permissionService.canAddParticipant(principal, 1L, "student")).isFalse();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canAddParticipantsManagerTeacher() {
		assertThat(permissionService.canAddParticipant(principal, 1L, "teacher")).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewStatusAdmin() {
		assertThat(permissionService.canViewStatus(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewStatusTeacher() {
		assertThat(permissionService.canViewStatus(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewStatusManager() {
		assertThat(permissionService.canViewStatus(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewStatusAssistant() {
		assertThat(permissionService.canViewStatus(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewStatusStudent() {
		assertThat(permissionService.canViewStatus(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canManageAssignmentsAdmin() {
		assertThat(permissionService.canManageAssignments(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canManageAssignmentsTeacher() {
		assertThat(permissionService.canManageAssignments(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canManageAssignmentsManager() {
		assertThat(permissionService.canManageAssignments(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canManageAssignmentsAssistant() {
		assertThat(permissionService.canManageAssignments(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canManageAssignmentsStudent() {
		assertThat(permissionService.canManageAssignments(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canEditCourseAdmin() {
		assertThat(permissionService.canEditCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canEditCourseTeacher() {
		assertThat(permissionService.canEditCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canEditCourseManager() {
		assertThat(permissionService.canEditCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canEditCourseAssistant() {
		assertThat(permissionService.canEditCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEditCourseStudent() {
		assertThat(permissionService.canEditCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canExportLabAdmin() {
		assertThat(permissionService.canExportLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canExportLabTeacher() {
		assertThat(permissionService.canExportLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canExportLabManager() {
		assertThat(permissionService.canExportLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canExportLabAssistant() {
		assertThat(permissionService.canExportLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canExportLabStudent() {
		assertThat(permissionService.canExportLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canRemoveCourseAdmin() {
		assertThat(permissionService.canRemoveCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canRemoveCourseTeacher() {
		assertThat(permissionService.canRemoveCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canRemoveCourseManager() {
		assertThat(permissionService.canRemoveCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canRemoveCourseAssistant() {
		assertThat(permissionService.canRemoveCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canRemoveCourseStudent() {
		assertThat(permissionService.canRemoveCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnrollCourseParticipates() {
		assertThat(permissionService.canEnrollCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canEnrollCourseAssists() {
		assertThat(permissionService.canEnrollCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canEnrollCourseManages() {
		assertThat(permissionService.canEnrollCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canEnrollCourseTeaches() {
		assertThat(permissionService.canEnrollCourse(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student100@tudelft.nl")
	@Transactional
	public void canEnrollCourse() {
		assertThat(permissionService.canEnrollCourse(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student99@tudelft.nl")
	@Transactional
	public void canEnrollCourseArchived() {
		Course course = new Course("archived", "archived", "", false, true);
		courseRepository.save(course);
		assertThat(permissionService.canEnrollCourse(principal, course.getId())).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewLabAdmin() {
		assertThat(permissionService.canViewLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewLabTeaches() {
		assertThat(permissionService.canViewLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewLabManages() {
		assertThat(permissionService.canViewLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewLabAssists() {
		assertThat(permissionService.canViewLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewLabParticipatesTrue() {
		assertThat(permissionService.canViewLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student100@tudelft.nl")
	@Transactional
	public void canViewLabParticipatesFalse() {
		assertThat(permissionService.canViewLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student100@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabDoesNotParticipates() {
		assertThat(permissionService.canEnqueueSelfForLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabEnqueueClosed() {
		Course course = courseRepository.findByIdOrThrow(1L);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now().plusDays(1),
				LocalDateTime.now().plusDays(1).plusHours(4)), course.getUsedRooms(), false);
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		labRepository.save(lab);
		assertThat(permissionService.canEnqueueSelfForLab(principal, lab.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabUserIsProcessed() {
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		Lab lab = labRepository.findByIdOrThrow(1L);
		Optional<Request> request = lab.getRequests().stream()
				.filter(r -> r.getRequestEntity().equals(userPrincipal.getUser())).findFirst();
		if (request.isPresent()) {
			int index = lab.getRequests().indexOf(request.get());
			Request r = lab.getRequests().get(index);
			r.setStatus(Request.Status.PROCESSING);
			lab.getRequests().set(index, r);
		}
		labRepository.save(lab);
		assertThat(permissionService.canEnqueueSelfForLab(principal, lab.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabNotOpen() {
		Course course = courseRepository.findByIdOrThrow(1L);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now(),
				LocalDateTime.now().plusHours(4)), course.getUsedRooms(), true);
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		lab.setEnqueueClosed(true);
		labRepository.save(lab);
		assertThat(permissionService.canEnqueueSelfForLab(principal, lab.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabUserNotAllowedMentorGroup() {
		Course course = new Course("groups", "groups", "", true);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now(),
				LocalDateTime.now().plusHours(4)), courseRepository.findByIdOrThrow(1L).getUsedRooms(),
				false);

		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		lab.setAllowWithoutMentorGroup(false);
		lab.setAllowedMentorGroups(
				new ArrayList<>(Collections.singletonList(new FirstYearMentorGroup("test"))));

		courseRepository.save(course);
		labRepository.save(lab);

		UserPrincipal userPrincipal = (UserPrincipal) principal;
		userPrincipal.getUser().addRole(new Student(userPrincipal.getUser(), course, LocalDateTime.now()));
		assertThat(permissionService.canEnqueueSelfForLab(principal, lab.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabUserAllowedMentorGroup() {
		Course course = new Course("groups", "groups", "", true);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now(),
				LocalDateTime.now().plusHours(4)), courseRepository.findByIdOrThrow(1L).getUsedRooms(),
				false);

		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		lab.setAllowWithoutMentorGroup(false);

		User student = userRepository.findById(1L).orElseThrow();
		FirstYearMentorGroup firstYearMentorGroup = new FirstYearMentorGroup("test");
		FirstYearStudent fys = new FirstYearStudent(student, firstYearMentorGroup);
		student.getFirstYearStudents().add(fys);
		firstYearMentorGroup.getStudents().add(fys);
		firstYearMentorGroup.setActive(true);
		lab.setAllowedMentorGroups(new ArrayList<>(Collections.singletonList(firstYearMentorGroup)));

		courseRepository.save(course);
		labRepository.save(lab);

		UserPrincipal userPrincipal = (UserPrincipal) principal;
		userPrincipal.getUser().addRole(new Student(userPrincipal.getUser(), course, LocalDateTime.now()));
		assertThat(permissionService.canEnqueueSelfForLab(principal, lab.getId())).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabFeedbackLab() {
		Course course = courseRepository.findByIdOrThrow(1L);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now(),
				LocalDateTime.now().plusHours(4)), courseRepository.findByIdOrThrow(1L).getUsedRooms(), true);
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		labRepository.save(lab);
		assertThat(permissionService.canEnqueueSelfForLab(principal, lab.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueSelfForLabTrue() {
		assertThat(permissionService.canEnqueueSelfForLab(principal, 2L)).isTrue();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canEnqueueOthersForLabAdmin() {
		assertThat(permissionService.canEnqueueOthersForLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canEnqueueOthersForLabTeacher() {
		assertThat(permissionService.canEnqueueOthersForLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canEnqueueOthersForLabManager() {
		assertThat(permissionService.canEnqueueOthersForLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canEnqueueOthersForLabAssistant() {
		assertThat(permissionService.canEnqueueOthersForLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEnqueueOthersForLabStudent() {
		assertThat(permissionService.canEnqueueOthersForLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canUnenqueueSelfFromLabStudent() {
		assertThat(permissionService.canUnenqueueSelfFromLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canUnenqueueSelfFromLabExamLabClosed() {
		Course course = courseRepository.findByIdOrThrow(1L);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now(),
				LocalDateTime.now().plusHours(4)), courseRepository.findByIdOrThrow(1L).getUsedRooms(),
				false);
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		lab.setExamLab(true);
		lab.setEnqueueClosed(true);
		labRepository.save(lab);
		assertThat(permissionService.canUnenqueueSelfFromLab(principal, lab.getId())).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canUnenqueueSelfFromLabExamLabOpen() {
		Course course = courseRepository.findByIdOrThrow(1L);
		Lab lab = new Lab(course, new LabSlot(LocalDateTime.now(),
				LocalDateTime.now().plusHours(4)), courseRepository.findByIdOrThrow(1L).getUsedRooms(),
				false);
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab.setAssignments(labRepository.findByIdOrThrow(1L).getAssignments());
		lab.setExamLab(true);
		lab.setEnqueueClosed(false);
		labRepository.save(lab);
		assertThat(permissionService.canUnenqueueSelfFromLab(principal, lab.getId())).isTrue();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canUnenqueueOthersFromLabAdmin() {
		assertThat(permissionService.canUnenqueuOthersFromLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canUnenqueueOthersFromLabTeacher() {
		assertThat(permissionService.canUnenqueuOthersFromLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canUnenqueueOthersFromLabManager() {
		assertThat(permissionService.canUnenqueuOthersFromLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canUnenqueueOthersFromLabAssistant() {
		assertThat(permissionService.canUnenqueuOthersFromLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canUnenqueueOthersFromLabStudent() {
		assertThat(permissionService.canUnenqueuOthersFromLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canNextRequestAdmin() {
		assertThat(permissionService.canNextRequest(principal)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canNextRequestTeacher() {
		assertThat(permissionService.canNextRequest(principal, 1L)).isTrue();
		assertThat(permissionService.canNextRequest(principal)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canNextRequestManager() {
		assertThat(permissionService.canNextRequest(principal, 1L)).isTrue();
		assertThat(permissionService.canNextRequest(principal)).isTrue();

	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canNextRequestAssistant() {
		assertThat(permissionService.canNextRequest(principal, 1L)).isTrue();
		assertThat(permissionService.canNextRequest(principal)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canNextRequestStudent() {
		assertThat(permissionService.canNextRequest(principal, 1L)).isFalse();
		assertThat(permissionService.canNextRequest(principal)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewRequestsAdmin() {
		assertThat(permissionService.canViewRequests(principal, 1L)).isTrue();
		assertThat(permissionService.canViewRequests(principal)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewRequestsTeacher() {
		assertThat(permissionService.canViewRequests(principal, 1L)).isTrue();
		assertThat(permissionService.canViewRequests(principal)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewRequestsManager() {
		assertThat(permissionService.canViewRequests(principal, 1L)).isTrue();
		assertThat(permissionService.canViewRequests(principal)).isTrue();

	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewRequestsAssistant() {
		assertThat(permissionService.canViewRequests(principal, 1L)).isTrue();
		assertThat(permissionService.canViewRequests(principal)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewRequestsStudent() {
		assertThat(permissionService.canViewRequests(principal, 1L)).isFalse();
		assertThat(permissionService.canViewRequests(principal)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canCreateLabAdmin() {
		assertThat(permissionService.canCreateLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canCreateLabTeacher() {
		assertThat(permissionService.canCreateLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canCreateLabManager() {
		assertThat(permissionService.canCreateLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canCreateLabAssistant() {
		assertThat(permissionService.canCreateLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canCreateLabStudent() {
		assertThat(permissionService.canCreateLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewRoomLayout() {
		assertThat(permissionService.canViewRoomLayout(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canEditLabAdmin() {
		assertThat(permissionService.canEditLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canEditLabTeacher() {
		assertThat(permissionService.canEditLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canEditLabManager() {
		assertThat(permissionService.canEditLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canEditLabAssistant() {
		assertThat(permissionService.canEditLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canEditLabStudent() {
		assertThat(permissionService.canEditLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canRemoveLabAdmin() {
		assertThat(permissionService.canRemoveLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canRemoveLabTeacher() {
		assertThat(permissionService.canRemoveLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canRemoveLabManager() {
		assertThat(permissionService.canRemoveLab(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canRemoveLabAssistant() {
		assertThat(permissionService.canRemoveLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canRemoveLabStudent() {
		assertThat(permissionService.canRemoveLab(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canManageLabSlotsAdmin() {
		assertThat(permissionService.canManageLabSlots(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canManageLabSlotsTeacher() {
		assertThat(permissionService.canManageLabSlots(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canManageLabSlotsManager() {
		assertThat(permissionService.canManageLabSlots(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canManageLabSlotsAssistant() {
		assertThat(permissionService.canManageLabSlots(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canManageLabSlotsStudent() {
		assertThat(permissionService.canManageLabSlots(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewNotificationTrue() {
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		Course course = courseRepository.findByIdOrThrow(1L);
		Notification notification = new Notification(userPrincipal.getUser(), course, "", "", 1);
		notificationRepository.save(notification);
		assertThat(permissionService.canViewNotification(principal, notification.getId())).isTrue();
	}

	@Test
	@WithUserDetails("student2@tudelft.nl")
	@Transactional
	public void canViewNotificationFalse() {
		User user = userRepository.findById(1L).orElseThrow();
		Course course = courseRepository.findByIdOrThrow(1L);
		Notification notification = new Notification(user, course, "", "", 1);
		notificationRepository.save(notification);
		assertThat(permissionService.canViewNotification(principal, notification.getId())).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewRequestAdmin() {
		assertThat(permissionService.canViewRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewRequestTeacher() {
		assertThat(permissionService.canViewRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewRequestManager() {
		assertThat(permissionService.canViewRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewRequestAssistant() {
		assertThat(permissionService.canViewRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewRequestStudentTrue() {
		assertThat(permissionService.canViewRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student2@tudelft.nl")
	@Transactional
	public void canViewRequestStudentFalse() {
		assertThat(permissionService.canViewRequest(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewFeedbackAdmin() {
		assertThat(permissionService.canViewFeedback(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewFeedbackTeacher() {
		assertThat(permissionService.canViewFeedback(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewFeedbackManager() {
		assertThat(permissionService.canViewFeedback(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewFeedbackAssistant() {
		assertThat(permissionService.canViewFeedback(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewFeedbackStudent() {
		assertThat(permissionService.canViewFeedback(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student2@tudelft.nl")
	@Transactional
	public void canViewFeedbackOtherStudent() {
		assertThat(permissionService.canViewFeedback(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewDeanonimizedFeedbackAdmin() {
		assertThat(permissionService.canViewDeanonimizedFeedback(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewDeanonimizedFeedbackTeacher() {
		assertThat(permissionService.canViewDeanonimizedFeedback(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewDeanonimizedFeedbackManager() {
		assertThat(permissionService.canViewDeanonimizedFeedback(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewDeanonimizedFeedbackAssistant() {
		assertThat(permissionService.canViewDeanonimizedFeedback(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewDeanonimizedFeedbackStudent() {
		assertThat(permissionService.canViewDeanonimizedFeedback(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewOwnFeedbackAdmin() {
		assertThat(permissionService.canViewOwnFeedback(principal)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewOwnFeedbackTeacher() {
		assertThat(permissionService.canViewOwnFeedback(principal)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewOwnFeedbackManager() {
		assertThat(permissionService.canViewOwnFeedback(principal)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewOwnFeedbackAssistant() {
		assertThat(permissionService.canViewOwnFeedback(principal)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewOwnFeedbackStudent() {
		assertThat(permissionService.canViewOwnFeedback(principal)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canViewRequestReasonAdmin() {
		assertThat(permissionService.canViewRequestReason(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canViewRequestReasonTeacher() {
		assertThat(permissionService.canViewRequestReason(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canViewRequestReasonManager() {
		assertThat(permissionService.canViewRequestReason(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canViewRequestReasonAssistant() {
		assertThat(permissionService.canViewRequestReason(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canViewRequestReasonStudent() {
		assertThat(permissionService.canViewRequestReason(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void canFinishRequestAdmin() {
		assertThat(permissionService.canFinishRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("teacher1@tudelft.nl")
	@Transactional
	public void canFinishRequestTeacher() {
		assertThat(permissionService.canFinishRequest(principal, 1L)).isTrue();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canFinishRequestManager() {
		Lab lab = labRepository.findByIdOrThrow(1L);
		User student = userRepository.findById(1L).orElseThrow();
		Request request = new Request(student, lab.getAssignments().get(0), lab.getRooms().get(0),
				lab.getAllowedRequestTypes().get(0), "", lab);
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		request.setAssistant(userPrincipal.getUser());
		requestRepository.save(request);
		assertThat(permissionService.canFinishRequest(principal, request.getId())).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canFinishRequestAssistant() {
		Lab lab = labRepository.findByIdOrThrow(1L);
		User student = userRepository.findById(1L).orElseThrow();
		Request request = new Request(student, lab.getAssignments().get(0), lab.getRooms().get(0),
				lab.getAllowedRequestTypes().get(0), "", lab);
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		request.setAssistant(userPrincipal.getUser());
		requestRepository.save(request);
		assertThat(permissionService.canFinishRequest(principal, request.getId())).isTrue();
	}

	@Test
	@WithUserDetails("student1@tudelft.nl")
	@Transactional
	public void canFinishRequestStudent() {
		assertThat(permissionService.canFinishRequest(principal, 1L)).isFalse();
	}

	@Test
	@WithUserDetails("manager1@tudelft.nl")
	@Transactional
	public void canFinishRequestManagerHandled() {
		Lab lab = labRepository.findByIdOrThrow(1L);
		User student = userRepository.findById(1L).orElseThrow();
		Request request = new Request(student, lab.getAssignments().get(0), lab.getRooms().get(0),
				lab.getAllowedRequestTypes().get(0), "", lab);
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		request.setAssistant(userPrincipal.getUser());
		request.setStatus(Request.Status.APPROVED);
		requestRepository.save(request);
		assertThat(permissionService.canFinishRequest(principal, request.getId())).isTrue();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canFinishRequestAssistantHandled() {
		Lab lab = labRepository.findByIdOrThrow(1L);
		User student = userRepository.findById(1L).orElseThrow();
		Request request = new Request(student, lab.getAssignments().get(0), lab.getRooms().get(0),
				lab.getAllowedRequestTypes().get(0), "", lab);
		UserPrincipal userPrincipal = (UserPrincipal) principal;
		request.setAssistant(userPrincipal.getUser());
		request.setStatus(Request.Status.APPROVED);
		requestRepository.save(request);
		assertThat(permissionService.canFinishRequest(principal, request.getId())).isFalse();
	}

	@Test
	@WithUserDetails("assistant1@tudelft.nl")
	@Transactional
	public void canFinishRequestWrongAssistant() {
		Lab lab = labRepository.findByIdOrThrow(1L);
		User student = userRepository.findById(1L).orElseThrow();
		User assistant = userRepository.findById(102L).orElseThrow();
		Request request = new Request(student, lab.getAssignments().get(0), lab.getRooms().get(0),
				lab.getAllowedRequestTypes().get(0), "", lab);
		request.setAssistant(assistant);
		requestRepository.save(request);
		assertThat(permissionService.canFinishRequest(principal, request.getId())).isFalse();
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void getUserException() {
		assertThrows(EntityNotFoundException.class, () -> permissionService.getUser(null));
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void getLabException() {
		assertThrows(EntityNotFoundException.class, () -> permissionService.getLab(10000000L));
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void getRoleException() {
		assertThrows(EntityNotFoundException.class, () -> permissionService.getRole(10000000L));
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void getNotificationException() {
		assertThrows(EntityNotFoundException.class, () -> permissionService.getNotification(10000000L));
	}

	@Test
	@WithUserDetails("admin@tudelft.nl")
	@Transactional
	public void getRequestException() {
		assertThrows(EntityNotFoundException.class, () -> permissionService.getRequest(10000000L));
	}
}
