/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.service;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.time.LocalDateTime;
import java.util.*;

import javax.transaction.Transactional;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;

import com.querydsl.core.types.Predicate;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
@Transactional
public class RequestServiceTest {

	@Autowired
	private RequestService requestService;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private AssignmentRepository assignmentRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private RoomRepository roomRepository;

	private User student, ta;
	private Lab lab;
	private Course course;

	private Predicate predicate;

	@BeforeEach
	public void setup() {
		course = courseRepository.findAll().get(0);
		lab = labRepository.findAll().get(0);

		course.addLab(lab);
		lab.setCourse(course);

		student = userRepository.findById(1L).orElseThrow();
		ta = new User("ta", "", "ta", "", DefaultRole.ROLE_ASSISTANT, 1);

		Role role = new Assistant(ta, course);
		ta.addRole(role);
		course.addRole(role);

		userRepository.save(ta);

		predicate = Mockito.mock(Predicate.class);
	}

	@Test
	public void createNotificationNull() {
		lab.setCommunicationMethod(null);
		assertThrows(NullPointerException.class, () -> RequestService.createNotification(lab, ta, student));
	}

	@Test
	public void createNotificationStudentTA() {
		createNotification(CommunicationMethod.STUDENT_VISIT_TA);
	}

	@Test
	public void createNotificationTAStudent() {
		createNotification(CommunicationMethod.TA_VISIT_STUDENT);
	}

	@Test
	public void createNotificationJitsi() {
		createNotification(CommunicationMethod.JITSI_MEET);
	}

	private void createNotification(CommunicationMethod cm) {
		lab.setCommunicationMethod(cm);
		Notification notification = RequestService.createNotification(lab, ta, student);

		assertThat(notification.getUser()).isEqualTo(student);
		assertThat(notification.getCourse()).isEqualTo(course);
	}

	@Test
	public void nextSimple() {
		Optional<Request> request = requestService.next(ta, lab, predicate);

		assertThat(request.isPresent()).isTrue();
		assertThat(request.get().getId()).isEqualTo(1L);
	}

	@Test
	public void nextTaProcessesRequest() {
		Request request = lab.getRequests().get(0);
		request.setStatus(Request.Status.PROCESSING);
		request.setAssistant(ta);
		lab.getRequests().set(0, request);
		labRepository.save(lab);

		Optional<Request> optionalRequest = requestService.next(ta, lab, predicate);
		assertThat(optionalRequest.isPresent()).isTrue();
		assertThat(optionalRequest.get().getId()).isEqualTo(request.getId());
	}

	@Test
	public void nextAssignedRequest() {
		Request request = lab.getRequests().get(0);
		request.setStatus(Request.Status.ASSIGNED);
		request.setAssistant(ta);
		lab.getRequests().set(0, request);
		labRepository.save(lab);

		Optional<Request> optionalRequest = requestService.next(ta, lab, predicate);
		assertThat(optionalRequest.isPresent()).isTrue();
		assertThat(optionalRequest.get().getId()).isEqualTo(request.getId());
	}

	@Test
	public void nextExamLab() {
		Lab temp = new Lab();
		temp.setCourse(course);
		temp.setCommunicationMethod(CommunicationMethod.TA_VISIT_STUDENT);
		temp.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		temp.setIntervalTime(15L);
		temp.setSignOffIntervals(true);
		temp.setExamLab(true);

		Assignment assignment = new Assignment(course, "assignment");
		assignmentRepository.save(assignment);
		temp.setAssignments(Collections.singletonList(assignment));
		RequestType requestType = new RequestType("Question");
		requestTypeRepository.save(requestType);
		temp.setAllowedRequestTypes(Collections.singletonList(requestType));
		Room room = new Room("room");
		roomRepository.save(room);
		temp.setRooms(Collections.singletonList(room));

		Request request = new Request();
		request.setStatus(Request.Status.PICKED);
		request.setSlot(new RequestSlot(LocalDateTime.now().minusMinutes(5), LocalDateTime.now()));
		request.setLab(temp);
		request.setAssignment(assignment);
		request.setRoom(room);
		request.setRequestType(requestType);
		request.setCreatedAt(LocalDateTime.now());
		requestRepository.save(request);

		Request request1 = new Request();
		request1.setStatus(Request.Status.PICKED);
		request1.setSlot(new RequestSlot(LocalDateTime.now().minusMinutes(5), LocalDateTime.now()));
		request1.setLab(temp);
		request1.setAssignment(assignment);
		request1.setRoom(room);
		request1.setRequestType(requestType);
		request1.setCreatedAt(LocalDateTime.now());
		requestRepository.save(request1);

		temp.setRequests(List.of(request, request1));
		labRepository.save(temp);

		Optional<Request> optionalRequest = requestService.next(ta, temp, predicate);
		assertThat(optionalRequest.isPresent()).isTrue();
	}

	@Test
	public void nextNoPendingRequestsExamLab() {
		Lab temp = new Lab();
		temp.setCourse(course);
		temp.setCommunicationMethod(CommunicationMethod.TA_VISIT_STUDENT);
		temp.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		temp.setIntervalTime(15L);
		temp.setSignOffIntervals(true);
		temp.setExamLab(true);

		Assignment assignment = new Assignment(course, "assignment");
		assignmentRepository.save(assignment);
		temp.setAssignments(Collections.singletonList(assignment));
		RequestType requestType = new RequestType("Question");
		requestTypeRepository.save(requestType);
		temp.setAllowedRequestTypes(Collections.singletonList(requestType));
		Room room = new Room("room");
		roomRepository.save(room);
		temp.setRooms(Collections.singletonList(room));

		labRepository.save(temp);

		Optional<Request> optionalRequest = requestService.next(ta, temp, predicate);
		assertThat(optionalRequest.isEmpty()).isTrue();
	}

	@Test
	public void nextNoPendingRequests() {
		Lab temp = new Lab();
		temp.setCourse(course);
		temp.setCommunicationMethod(CommunicationMethod.TA_VISIT_STUDENT);
		temp.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now()));
		temp.setIntervalTime(15L);
		temp.setSignOffIntervals(true);

		Assignment assignment = new Assignment(course, "assignment");
		assignmentRepository.save(assignment);
		temp.setAssignments(Collections.singletonList(assignment));
		RequestType requestType = new RequestType("Question");
		requestTypeRepository.save(requestType);
		temp.setAllowedRequestTypes(Collections.singletonList(requestType));
		Room room = new Room("room");
		roomRepository.save(room);
		temp.setRooms(Collections.singletonList(room));

		labRepository.save(temp);

		Optional<Request> optionalRequest = requestService.next(ta, temp, predicate);
		assertThat(optionalRequest.isEmpty()).isTrue();
	}

	@Test
	public void revokeNonPendingRequest() {
		Request request = lab.getRequests().get(0);
		request.setStatus(Request.Status.PROCESSING);

		assertThrows(IllegalStateException.class, () -> requestService.revoke(request));
	}

	@Test
	public void revokePendingRequest() {
		Request request = lab.getRequests().get(0);
		request.setStatus(Request.Status.PENDING);

		int size = lab.getPending().size();
		requestService.revoke(request);

		assertThat(lab.getPending().size()).isEqualTo(size - 1);
	}

	@Test
	public void approveNoRequestProcessingEvent() {
		Request request = lab.getRequests().get(0);
		int eventSize = request.getEvents().size();

		requestService.approve(ta, request, "");

		assertThat(request.getEvents().size()).isEqualTo(eventSize + 2);
	}

	@Test
	public void approveTwice() {
		Request request = lab.getRequests().get(0);
		int eventSize = request.getEvents().size();

		requestService.approve(ta, request, "");
		assertThat(request.getEvents().size()).isEqualTo(eventSize + 2);

		requestService.approve(ta, request, "");
		assertThat(request.getEvents().size()).isEqualTo(eventSize + 3);
	}

	@Test
	public void forwardNonPrivilegedAssistant() {
		Request request = new Request();
		request.setLab(lab);

		assertThrows(IllegalArgumentException.class, () -> requestService.forward(request, new User(), ""));
	}

	@Test
	public void forwardDifferentAssistant() {
		Request request = new Request();
		request.setAssistant(new User());
		request.setLab(lab);
		request.setAssignment(lab.getAssignments().get(0));
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));

		requestService.forward(request, ta, "");
	}

	@Test
	public void forwardNoAssistant() {
		Request request = new Request();
		request.setAssistant(null);
		request.setLab(lab);
		request.setAssignment(lab.getAssignments().get(0));
		request.setRoom(lab.getRooms().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));

		requestService.forward(request, ta, "");
	}

	@Test
	public void forwardSameAssistant() {
		Request request = new Request();
		request.setAssistant(ta);
		request.setLab(lab);

		assertThrows(IllegalArgumentException.class, () -> requestService.forward(request, ta, ""));
	}

	@Test
	public void setSubmissionUrlNoUser() {
		Request request = new Request();
		request.setRequestEntity(new Group());

		assertThat(requestService.setSubmissionUrl(request)).isEqualTo("");
	}

	@Test
	public void setSubmissionUrlNoSubmissionUrl() {
		Request request = new Request();
		request.setRequestEntity(student);
		request.setLab(lab);
		course.setSubmissionUrl(null);

		assertThat(requestService.setSubmissionUrl(request)).isEqualTo("");
	}

	@Test
	public void setSubmissionUrlEmptySubmissionUrl() {
		Request request = new Request();
		request.setRequestEntity(student);
		request.setLab(lab);
		course.setSubmissionUrl("");

		assertThat(requestService.setSubmissionUrl(request)).isEqualTo("");
	}

	@Test
	public void setSubmissionUrlSimple() {
		Request request = new Request();
		request.setRequestEntity(student);
		request.setLab(lab);
		course.setSubmissionUrl("submissionUrl");

		assertThat(requestService.setSubmissionUrl(request)).isEqualTo("submissionUrl");
	}

	@Test
	public void getProcessingRequestEdgeCases() {
		Request nullAssistant = new Request();
		nullAssistant.setStatus(Request.Status.PROCESSING);
		nullAssistant.setAssistant(null);
		nullAssistant.setCreatedAt(LocalDateTime.now());

		Request wrongAssistant = new Request();
		wrongAssistant.setStatus(Request.Status.PROCESSING);
		wrongAssistant.setAssistant(student);
		wrongAssistant.setCreatedAt(LocalDateTime.now());

		Lab lab = new Lab();
		lab.addRequest(nullAssistant);
		lab.addRequest(wrongAssistant);
		lab.setId(12321L);

		assertThat(requestService.next(ta, lab, predicate)).isEmpty();
	}

	@Test
	public void getAssignedRequestEdgeCases() {
		Request nullAssistant = new Request();
		nullAssistant.setStatus(Request.Status.ASSIGNED);
		nullAssistant.setAssistant(null);
		nullAssistant.setCreatedAt(LocalDateTime.now());

		Request wrongAssistant = new Request();
		wrongAssistant.setStatus(Request.Status.ASSIGNED);
		wrongAssistant.setAssistant(student);
		wrongAssistant.setCreatedAt(LocalDateTime.now());

		Lab lab = new Lab();
		lab.addRequest(nullAssistant);
		lab.addRequest(wrongAssistant);
		lab.setId(12321L);

		assertThat(requestService.next(ta, lab, predicate)).isEmpty();
	}

	@Test
	public void updateRemainingPositionsEdgeCases() {
		Request request = new Request();
		request.setRequestEntity(student);
		request.setRoom(lab.getRooms().get(0));
		request.setAssignment(lab.getAssignments().get(0));
		request.setRequestType(lab.getAllowedRequestTypes().get(0));
		request.setCreatedAt(LocalDateTime.now());
		request.setStatus(Request.Status.PENDING);

		Request noUser = new Request();
		noUser.setStatus(Request.Status.PENDING);
		noUser.setRequestEntity(new Group());
		noUser.setCreatedAt(LocalDateTime.now());

		Request sameUser = new Request();
		sameUser.setStatus(Request.Status.PENDING);
		sameUser.setRequestEntity(student);
		sameUser.setCreatedAt(LocalDateTime.now());

		Lab temp = new Lab();
		temp.addRequest(request);
		temp.addRequest(noUser);
		temp.addRequest(sameUser);

		int size = temp.getPending().size();
		requestService.revoke(request);

		assertThat(temp.getPending().size()).isEqualTo(size - 1);
	}

}
