/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.task;

import static org.assertj.core.api.AssertionsForInterfaceTypes.assertThat;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;

import nl.tudelft.ewi.queue.QueueApplication;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;
import nl.tudelft.ewi.queue.service.LabService;
import nl.tudelft.ewi.queue.tasks.RequestTasks;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.transaction.annotation.Transactional;

@SpringBootTest(classes = QueueApplication.class)
@WebAppConfiguration
@AutoConfigureMockMvc
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
@Transactional
public class RequestTaskTest {

	private Lab lab;

	LocalDateTime startOfLab;

	@Autowired
	private LabService labService;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private AssignmentRepository assignmentRepository;

	@Autowired
	private RoomRepository roomRepository;

	@Autowired
	private LabRepository labRepository;

	private Room room;
	private Assignment assignment;
	private RequestType requestType;
	List<Role> students;

	@Autowired
	private RequestTasks requestTask;

	Lab lab2;

	@BeforeEach
	public void init() {
		startOfLab = LocalDateTime.now().plusMinutes(1);
		lab = new Lab();
		lab.setExamLab(true);
		lab.setExamLabPercentage(10);
		LabSlot labSlot = new LabSlot(startOfLab, startOfLab.plusHours(4));
		lab.setSlot(labSlot);
		lab.setIntervalTime(10L);
		lab.setSignOffIntervals(true);
		lab.setCapacity(2L);
		Course course = courseRepository.findByIdOrThrow(1L);
		course.setLabs(Collections.singletonList(lab));
		students = course.getStudents();
		room = roomRepository.findAll().get(0);
		assignment = assignmentRepository.findAll().iterator().next();
		requestType = requestTypeRepository.findAll().iterator().next();
		lab.setCourse(course);
		lab.setRooms(Collections.singletonList(room));
		lab.setAssignments(Collections.singletonList(assignment));
		lab.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		labRepository.save(lab);

		lab2 = new Lab();
		lab2.setExamLab(true);
		lab2.setExamLabPercentage(10);
		LabSlot labSlot2 = new LabSlot(startOfLab, startOfLab.plusHours(1));
		lab2.setSlot(labSlot2);
		lab2.setIntervalTime(30L);
		lab2.setSignOffIntervals(true);
		lab2.setCapacity(20L);
		Course course2 = courseRepository.findByIdOrThrow(1L);
		course2.setLabs(Collections.singletonList(lab2));
		students = course.getStudents();
		room = roomRepository.findAll().get(0);
		assignment = assignmentRepository.findAll().iterator().next();
		requestType = requestTypeRepository.findAll().iterator().next();
		lab2.setCourse(course);
		lab2.setRooms(Collections.singletonList(room));
		lab2.setAssignments(Collections.singletonList(assignment));
		lab2.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		labRepository.save(lab2);
	}

	@Test
	public void testWith1Enqueue() {
		enqueueXamount(1, lab);
		requestTask.randomSelectForExamLab();
		int numberOfRequests = lab.getRequests().size();
		long numberOfPicked = lab.getRequests().stream().filter(Request::isPicked).count();
		assertThat(numberOfPicked).isEqualTo(numberOfRequests);
		assertThat(lab.isEnqueueClosed()).isTrue();
	}

	@Test
	public void testWith10Enqueue() {
		enqueueXamount(10, lab);
		requestTask.randomSelectForExamLab();
		long numberOfPicked = lab.getRequests().stream().filter(Request::isPicked).count();
		assertThat(numberOfPicked).isGreaterThanOrEqualTo(1);
		assertThat(lab.isEnqueueClosed()).isTrue();
	}

	@Test
	public void testWith12Enqueue() {
		enqueueXamount(12, lab);
		requestTask.randomSelectForExamLab();
		long numberOfPicked = lab.getRequests().stream().filter(Request::isPicked).count();
		System.out.println("numberPicked" + numberOfPicked);
		assertThat(numberOfPicked).isGreaterThanOrEqualTo(2);
		assertThat(lab.isEnqueueClosed()).isTrue();
	}

	@Test
	public void testWith25Enqueue() {
		enqueueXamount(25, lab);
		requestTask.randomSelectForExamLab();
		long numberOfPicked = lab.getRequests().stream().filter(Request::isPicked).count();
		System.out.println("numberPicked" + numberOfPicked);
		assertThat(numberOfPicked).isGreaterThanOrEqualTo(3);
		assertThat(lab.isEnqueueClosed()).isTrue();
	}

	@Test
	public void testWith25EnqueueSmallLab() {
		enqueueXamount(25, lab2);
		requestTask.randomSelectForExamLab();
		long numberOfPicked = lab2.getRequests().stream().filter(Request::isPicked).count();
		System.out.println("numberPicked " + numberOfPicked);
		assertThat(numberOfPicked).isGreaterThanOrEqualTo(3);
		assertThat(lab2.isEnqueueClosed()).isTrue();
	}

	private void enqueueXamount(int x, Lab lab) {
		for (int i = 0; i < x; i++) {
			int slotNumber = Math.floorDiv(i, lab.getCapacity().intValue());
			Request request = new Request();
			RequestSlot requestSlot = new RequestSlot(startOfLab.plusMinutes(10 * slotNumber),
					startOfLab.plusMinutes((10 * slotNumber) + 10));
			request.setSlot(requestSlot);
			request.setRequestEntity(students.get(i).getUser());
			request.setLab(lab);
			request.setRoom(room);
			request.setAssignment(assignment);
			request.setRequestType(requestType);
			labService.enqueue(request);
		}
	}

}
