/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.viewmodel;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.when;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;

import nl.tudelft.ewi.queue.model.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

public class RequestCreateModelTest {

	private RequestCreateModel requestCreateModel, other;

	@BeforeEach
	public void init() {
		requestCreateModel = new RequestCreateModel();
		other = new RequestCreateModel();

		RequestSlot requestSlot = new RequestSlot(LocalDateTime.of(2020, 1, 1, 0, 0),
				LocalDateTime.of(2020, 1, 1, 1, 0));
		requestSlot.setId(1L);
		requestCreateModel.setSlot(requestSlot);
		other.setSlot(requestSlot);

		Room room = new Room();
		room.setId(1L);
		requestCreateModel.setRoom(room);
		other.setRoom(room);

		requestCreateModel.setJitsi(false);
		other.setJitsi(false);

		User user = Mockito.mock(User.class);
		when(user.getId()).thenReturn(1L);
		requestCreateModel.setRequestEntity(user);
		other.setRequestEntity(user);

		RequestType requestType = new RequestType("request type");
		requestCreateModel.setRequestType(requestType);
		other.setRequestType(requestType);

		requestCreateModel.setComment("comment");
		other.setComment("comment");

		requestCreateModel.setQuestion("question");
		other.setQuestion("question");

		Lab lab = new Lab();
		lab.setId(1L);
		requestCreateModel.setLab(lab);
		other.setLab(lab);

		Assignment assignment = new Assignment(new Course(), "assignment");
		requestCreateModel.setAssignment(assignment);
		other.setAssignment(assignment);

		lab.addAssignment(assignment);
		lab.addRoom(room);
		RequestType requestTypeQuestion = new RequestType("question");
		lab.setAllowedRequestTypes(List.of(requestType, requestTypeQuestion));
	}

	@Test
	public void hasErrorsFalse() {
		assertThat(requestCreateModel.hasErrors()).isFalse();
	}

	@Test
	public void hasErrorsLabNull() {
		requestCreateModel.setLab(null);
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsRequestEntityNull() {
		requestCreateModel.setRequestEntity(null);
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsNoAllowedAssignments() {
		requestCreateModel.getLab().setAssignments(Collections.emptyList());
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsNoAllowedRooms() {
		requestCreateModel.getLab().setRooms(Collections.emptyList());
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsNoAllowedRequestTypes() {
		requestCreateModel.getLab().setAllowedRequestTypes(Collections.emptyList());
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsQuestionNull() {
		requestCreateModel.setQuestion(null);
		RequestType requestType = new RequestType("question");
		requestCreateModel.setRequestType(requestType);
		requestCreateModel.getLab().setAllowedRequestTypes(List.of(requestType));
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsQuestionTooShort() {
		requestCreateModel.setQuestion("");
		RequestType requestType = new RequestType("question");
		requestCreateModel.setRequestType(requestType);
		requestCreateModel.getLab().setAllowedRequestTypes(List.of(requestType));
		assertThat(requestCreateModel.hasErrors()).isTrue();
	}

	@Test
	public void hasErrorsQuestionCorrect() {
		requestCreateModel.setQuestion("question question question");
		RequestType requestType = new RequestType("question");
		requestCreateModel.setRequestType(requestType);
		requestCreateModel.getLab().setAllowedRequestTypes(List.of(requestType));
		assertThat(requestCreateModel.hasErrors()).isFalse();
	}

	@Test
	public void equalsThis() {
		assertThat(requestCreateModel.equals(requestCreateModel)).isTrue();
	}

	@Test
	public void equalsOther() {
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsOtherInstance() {
		assertThat(requestCreateModel.equals(new Course())).isFalse();
	}

	@Test
	public void equalsThisSlotNull() {
		requestCreateModel.setSlot(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothSlotNull() {
		requestCreateModel.setSlot(null);
		other.setSlot(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentSlot() {
		requestCreateModel.setSlot(new RequestSlot(LocalDateTime.now(), LocalDateTime.now()));
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisRoomNull() {
		requestCreateModel.setRoom(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothRoomNull() {
		requestCreateModel.setRoom(null);
		other.setRoom(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentRoom() {
		requestCreateModel.setRoom(new Room());
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisJitsiNull() {
		requestCreateModel.setJitsi(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothJitsiNull() {
		requestCreateModel.setJitsi(null);
		other.setJitsi(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentJitsi() {
		requestCreateModel.setJitsi(true);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisRequestEntityNull() {
		requestCreateModel.setRequestEntity(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothRequestEntityNull() {
		requestCreateModel.setRequestEntity(null);
		other.setRequestEntity(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentRequestEntity() {
		requestCreateModel.setRequestEntity(new Group());
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisRequestTypeNull() {
		requestCreateModel.setRequestType(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothRequestTypeNull() {
		requestCreateModel.setRequestType(null);
		other.setRequestType(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentRequestType() {
		requestCreateModel.setRequestType(new RequestType(""));
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisCommentNull() {
		requestCreateModel.setComment(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothCommentNull() {
		requestCreateModel.setComment(null);
		other.setComment(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentComment() {
		requestCreateModel.setComment("");
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisQuestionNull() {
		requestCreateModel.setQuestion(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothQuestionNull() {
		requestCreateModel.setQuestion(null);
		other.setQuestion(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentQuestion() {
		requestCreateModel.setQuestion("");
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisLabNull() {
		requestCreateModel.setLab(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothLabNull() {
		requestCreateModel.setLab(null);
		other.setLab(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentLab() {
		requestCreateModel.setLab(new Lab());
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsThisAssignmentNull() {
		requestCreateModel.setAssignment(null);
		assertThat(requestCreateModel.equals(other)).isFalse();
	}

	@Test
	public void equalsBothAssignmentNull() {
		requestCreateModel.setAssignment(null);
		other.setAssignment(null);
		assertThat(requestCreateModel.equals(other)).isTrue();
		assertThat(requestCreateModel.hashCode()).isEqualTo(other.hashCode());
	}

	@Test
	public void equalsDifferentAssignment() {
		requestCreateModel.setAssignment(new Assignment());
		assertThat(requestCreateModel.equals(other)).isFalse();
	}
}
