import com.diffplug.gradle.spotless.SpotlessExtension
import nl.javadude.gradle.plugins.license.DownloadLicensesExtension
import nl.javadude.gradle.plugins.license.LicenseExtension
import org.springframework.boot.gradle.tasks.run.BootRun

group = "nl.tudelft.ewi.queue"
version = "2.0.0"

val javaVersion = JavaVersion.VERSION_11

val libradorVersion = "1.0.3-SNAPSHOT6"
val labradoorVersion = "1.0.7-SNAPSHOT9"

val springBootVersion = "2.3.8.RELEASE"
val springSecTestVersion = "5.3.3.RELEASE"

val thymeleafVersion = "3.0.9.RELEASE"
val thymeleafLayoutDialectVersion = "2.3.0"

val lombokVersion = "1.18.12"

val queryDslVersion = "4.3.1"
val jacksonVersion = "2.10.3"
val hibernateVersion = "5.4.24.Final"

// A definition of all dependencies and repositories where to find them that need to
// be available during compilation of this build script.
buildscript {
    dependencies {
        // Spring Loaded for hot reloading Java classes within IntelliJ
        // specifically for the Spring Boot development capabilities.
        classpath("org.springframework", "springloaded", "1.2.8.RELEASE")
    }
}

// The repositories used to lookup dependencies.
repositories {
    mavenLocal()
    mavenCentral()

    maven {
        url = uri("https://build.shibboleth.net/nexus/content/repositories/releases")
    }
    maven {
        name = "GitLab Librador repository"
        url = uri("https://gitlab.ewi.tudelft.nl/api/v4/projects/3634/packages/maven")
    }
    maven {
        name = "GitLab Labradoor repository"
        url = uri("https://gitlab.ewi.tudelft.nl/api/v4/projects/3611/packages/maven")
    }
}

// The plugins used by Gradle to generate files, start Spring boot, perform static analysis etc.
plugins {
    // Plugin for the Kotlin-DSL to be on classpath, disabled
    // because it does not need to be applied during build.
    `kotlin-dsl` apply false

    // Standard plugins for Gradle to work properly
    java
    `maven-publish`
    eclipse
    idea
    jacoco

    // Spring plugins for managing dependencies and creating
    // a nice Spring Boot application.
    id("org.springframework.boot").version("2.3.3.RELEASE")
    id("io.spring.dependency-management").version("1.0.9.RELEASE")

    // Plugin to provide task to check the current versions of
    // dependencies and of Gradle to see if updates are available.
    // TODO: Check if this works after upgrade 0.17.0 -> 0.28.0
    id("com.github.ben-manes.versions").version("0.28.0")

    // Spotless plugin for checking style of Java code.
    // TODO: Check if this works after upgrade 3.24.2 -> 3.27.2
    id("com.diffplug.gradle.spotless").version("3.27.2")

    // Plugin for checking license headers within our code and files.
    id("com.github.hierynomus.license").version("0.15.0")

    // Plugin for checking security issues in dependencies of this project.
    id("org.owasp.dependencycheck").version("5.3.0") apply false

    // Open API generator for generating the Client code for Labracore.
    id("org.openapi.generator").version("4.2.3")

    // Sass compiler plugin
    id("org.kravemir.gradle.sass").version("1.2.2")
}

sourceSets {
    main {
        java {
            srcDir(file("src/main/java"))
        }

    }

    test {
        java {
            srcDir(file("src/test/java"))
        }
    }
}

/////// Plugins configurations ///////
java {
    sourceCompatibility = javaVersion
    targetCompatibility = javaVersion
}

publishing {
    publications {
        create<MavenPublication>("generatePom") {
            from(components.findByName("java"))
            pom {
                withXml {
                    val repos = asNode().appendNode("repositories")
                    fun repository(id: String, url: String) {
                        val repo = repos.appendNode("repository")
                        repo.appendNode("id", id)
                        repo.appendNode("url", url)
                    }

                    repository("shibboleth", "https://build.shibboleth.net/nexus/content/repositories/releases")
                    repository("librador", "https://gitlab.ewi.tudelft.nl/api/v4/projects/3634/packages/maven")
                    repository("labradoor", "https://gitlab.ewi.tudelft.nl/api/v4/projects/3611/packages/maven")
                }
            }
        }
    }
}

// Configure Jacoco testing plugin.
configure<JacocoPluginExtension> {
    toolVersion = "0.8.5"
}

// Configure license plugins.
configure<DownloadLicensesExtension> {
    includeProjectDependencies = true
}

configure<LicenseExtension> {
    header = file("$rootDir/LICENSE.header")
    skipExistingHeaders = false

    mapping(mapOf(
            "java" to "SLASHSTAR_STYLE",
            "template" to "SCRIPT_STYLE"
    ))

    excludes(listOf(
            "**/*.json",
            "**/*.scss",
            "**/*.ico",
            "**/*.png",
            "**/*.jks",
            "**/*.css"
    ))
}

// Configure Spotless plugin for style checking Java code.
configure<SpotlessExtension> {
    java {
        // Use the eclipse formatter format and import order.
        eclipse().configFile(file("eclipse-formatter.xml"))
        importOrderFile(file("$rootDir/importorder.txt"))

        // Check for a license header in the form of LICENSE.header.java.
        licenseHeaderFile(file("$rootDir/LICENSE.header.java"))

        // Default added rules.
        paddedCell()
        removeUnusedImports()
        trimTrailingWhitespace()
        endWithNewline()
    }
}

/////// TASKS ///////
val jacocoTestReport by tasks.getting(JacocoReport::class) {
    group = "Reporting"
    reports {
        xml.isEnabled = true
        csv.isEnabled = true

        html.destination = file("$buildDir/reports/coverage")
    }
}

val jar by tasks.getting(Jar::class) {
    // Set the name and version of the produced JAR
    archiveBaseName.set("queue")
    archiveVersion.set("1.0.0")
}

// Configure Spring Boot plugin task for running the application.
val bootRun by tasks.getting(BootRun::class) {
    sourceResources(sourceSets.main.get())
}

// Configure the sass compiling task to use the sass and css directories.
val sassCompile = tasks.create("sassCompile", org.kravemir.gradle.sass.SassCompileTask::class) {
    srcDir = file("src/main/resources/static/sass")
    outDir = file("src/main/resources/static/css")
}

tasks.named("processResources") {
    dependsOn(sassCompile)
}

tasks.withType<Test>().configureEach {
    useJUnitPlatform()
    minHeapSize = "256m"
    maxHeapSize = "1024m"
}

dependencies {
    /////// Spring dependencies ///////

    // Generic Spring Boot starter dependencies
    implementation("org.springframework.boot", "spring-boot-starter-data-jpa", springBootVersion)
    implementation("org.springframework.boot", "spring-boot-starter-data-rest", springBootVersion)
    implementation("org.springframework.boot", "spring-boot-starter-web", springBootVersion)
    implementation("org.springframework.boot", "spring-boot-starter-websocket", springBootVersion)
    implementation("org.springframework.boot", "spring-boot-starter-webflux", springBootVersion)
    implementation("org.springframework.boot", "spring-boot-starter-validation", springBootVersion)

    // Spring Boot and jedis dependencies for interacting with redis session stores.
    implementation("redis.clients", "jedis", "2.9.0")
    implementation("org.springframework.boot", "spring-boot-starter-data-redis", springBootVersion)
    implementation("org.springframework.session", "spring-session-core", "2.3.1.RELEASE")
    implementation("org.springframework.session", "spring-session-data-redis", "2.3.1.RELEASE")

    // Dependencies for enabling Spring security + SAML security in Spring
    implementation("org.springframework.boot", "spring-boot-starter-security", springBootVersion)
    implementation("org.springframework.security.extensions", "spring-security-saml2-core", "1.0.10.RELEASE")

    // Dependency for websocket security by Spring
    implementation("org.springframework.security", "spring-security-messaging", "5.4.0")

    // Jaeger Spring dependencies for creating and sending Spans
    implementation("io.opentracing.contrib", "opentracing-spring-jaeger-starter", "3.1.2")
    implementation("io.opentracing.contrib", "opentracing-spring-jaeger-web-starter", "3.1.2")

    // Dependencies for enabling Thymeleaf templating language with Spring.
    implementation("org.springframework.boot", "spring-boot-starter-thymeleaf", springBootVersion)
    implementation("org.thymeleaf", "thymeleaf-spring5", "3.0.11.RELEASE")
    implementation("org.thymeleaf.extras", "thymeleaf-extras-springsecurity5", "3.0.4.RELEASE")
    implementation("org.thymeleaf.extras", "thymeleaf-extras-java8time", "3.0.4.RELEASE")
    implementation("nz.net.ultraq.thymeleaf", "thymeleaf-layout-dialect", "2.4.1")

    /////// Other dependencies ///////

    // Labrador project dependencies
    implementation("nl.tudelft.labrador", "librador", libradorVersion) {
        exclude("org.springframework.boot", "spring-boot-starter-data-rest")
    }
    implementation("nl.tudelft.labrador", "labradoor", labradoorVersion)

    // Java assist dependency
    implementation("org.javassist", "javassist", "3.25.0-GA")

    //    implementation("org.apache.httpcomponents", "fluent-hc", "4.5.5")

    // Bouncycastle for implementations of the Java Crypto API (JDK1.5-11)
    implementation("org.bouncycastle", "bcpkix-jdk15on", "1.64")
    implementation("org.bouncycastle", "bcprov-jdk15on", "1.64")

    // WebPush library for sending push notifications
    implementation("nl.martijndwars", "web-push", "5.1.0")

    //Jackson + JSON for parsing and (de-)serializing JSON objects
    implementation("org.json", "json", "20190722")
    implementation("com.fasterxml.jackson.core", "jackson-core", jacksonVersion)
    implementation("com.fasterxml.jackson.core", "jackson-databind", jacksonVersion)
    implementation("com.fasterxml.jackson.dataformat", "jackson-dataformat-csv", jacksonVersion)
    implementation("com.fasterxml.jackson.module", "jackson-modules-java8", jacksonVersion, ext = "pom")

    // Better Streams API than Java
    implementation("org.jooq", "jool", "0.9.12")

    // Apache commons for many helpful utility classes
    implementation("org.apache.commons", "commons-lang3", "3.0")

    // QueryDSL for providing a DSL based on Entity classes to write database queries
    implementation("com.querydsl", "querydsl-jpa", queryDslVersion)

    // Database migration + database driver dependencies
    implementation("org.liquibase", "liquibase-core")
    implementation("com.h2database", "h2")
    implementation("mysql", "mysql-connector-java", "8.0.18")
    implementation("org.mariadb.jdbc", "mariadb-java-client", "2.1.2")
    implementation("org.postgresql", "postgresql", "9.4.1212")

    // EE XML parsing and deserialization dependencies
    //    implementation("javax.xml.bind", "jaxb-api", "2.3.0")
    //    implementation("org.glassfish.jaxb", "jaxb-runtime", "2.3.0")

    // EE Jakarta
    implementation("com.sun.activation", "jakarta.activation", "1.2.1")
    implementation("com.sun.activation", "jakarta.activation", "1.2.1")
    implementation("com.sun.mail", "jakarta.mail", "1.6.4")

    // Sentry for writing error logs to a server for developer access
    implementation("io.sentry", "sentry-spring", "1.7.30")

    // Flexmark for parsing Markdown into HTML
    implementation("com.vladsch.flexmark", "flexmark-all", "0.50.42")

    // International Unicode expansion for ordinal strings
    implementation("com.ibm.icu", "icu4j", "67.1")

    // Dependency for mapping one data class to another
    implementation("org.modelmapper", "modelmapper", "2.3.6")

    // Webjars to be loaded within HTML resources
    implementation("org.webjars", "webjars-locator-core", "0.43")
    implementation("org.webjars", "jquery", "3.5.1")
    implementation("org.webjars", "bootstrap", "4.3.1")
    implementation("org.webjars", "bootstrap-select", "1.13.8")
    implementation("org.webjars", "font-awesome", "5.10.1")
    implementation("org.webjars", "stomp-websocket", "2.3.3-1")
    implementation("org.webjars", "handlebars", "4.0.14")
    implementation("org.webjars", "chartjs", "2.7.0")
    implementation("org.webjars", "tempusdominus-bootstrap-4", "5.1.2")
    implementation("org.webjars.bower", "momentjs", "2.20.1")
    implementation("org.webjars", "fullcalendar", "5.6.0")

    /////// Test dependencies ///////
    testImplementation("org.springframework.boot", "spring-boot-starter-test", springBootVersion)
    testImplementation("org.springframework.security", "spring-security-test", springSecTestVersion)

    /////// Annotation processing dependencies ///////
    annotationProcessor("javax.annotation", "javax.annotation-api", "1.3.2")

    annotationProcessor("org.springframework.boot", "spring-boot-configuration-processor", springBootVersion)
    annotationProcessor("org.hibernate.javax.persistence", "hibernate-jpa-2.1-api", "1.0.2.Final")
    annotationProcessor("com.querydsl", "querydsl-apt", queryDslVersion, classifier = "jpa")

    compileOnly("org.projectlombok", "lombok", lombokVersion)
    testCompileOnly("org.projectlombok", "lombok", lombokVersion)
    annotationProcessor("org.projectlombok", "lombok", lombokVersion)
    testAnnotationProcessor("org.projectlombok", "lombok", lombokVersion)
}

