/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue;

import static nl.tudelft.queue.model.enums.RequestType.QUESTION;
import static nl.tudelft.queue.model.enums.RequestType.SUBMISSION;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import nl.tudelft.labracore.api.*;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.dto.create.RequestCreateDTO;
import nl.tudelft.queue.dto.create.labs.ExamLabCreateDTO;
import nl.tudelft.queue.dto.create.labs.RegularLabCreateDTO;
import nl.tudelft.queue.dto.id.TimeSlotIdDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.embeddables.ExamLabConfig;
import nl.tudelft.queue.model.embeddables.Slot;
import nl.tudelft.queue.model.embeddables.SlottedLabConfig;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.service.LabService;
import nl.tudelft.queue.service.RequestService;
import nl.tudelft.queue.service.RoleDTOService;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.DependsOn;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Service;

@Service
@Profile("development")
@DependsOn({
		"nl.tudelft.librador.SpringContext"
})
public class DevDatabaseLoader implements InitializingBean {
	@Autowired
	private DbLoaderControllerApi dbApi;

	@Autowired
	private EditionControllerApi eApi;

	@Autowired
	private RoomControllerApi rApi;

	@Autowired
	private ModuleControllerApi mApi;

	@Autowired
	private LabService ls;

	@Autowired
	private RequestService rs;

	@Autowired
	private RoleDTOService rds;

	private List<RoomSummaryDTO> rooms;
	private List<PersonSummaryDTO> people;
	private List<EditionDetailsDTO> editions;

	private EditionDetailsDTO oop20_21;
	private EditionDetailsDTO ads20_21;

	private ModuleDetailsDTO oop20_21ModuleA;
	private ModuleDetailsDTO ads20_21ModuleA;
	private ModuleDetailsDTO ads20_21ModuleI;

	private List<PersonSummaryDTO> oopStudents;
	private List<PersonSummaryDTO> adsStudents;

	private Lab oopL1;
	private Lab oopL2;
	private Lab oopE;

	private Lab adsL1I;
	private Lab adsL1A;
	private Lab adsL2I;

	@Override
	@Transactional
	public synchronized void afterPropertiesSet() throws Exception {
		fetch();

		initLabs();
		initRequests();
	}

	private void fetch() {
		rooms = rApi.getAllRooms().collectList().block();
		people = dbApi.getAllPeopleInDB().collectList().block();
		editions = eApi.getAllEditions().collectList().block();

		oop20_21 = eApi.getEditionById(editions.stream()
				.filter(e -> "OOP NOW".equals(e.getName())).findFirst().get().getId()).block();
		ads20_21 = eApi.getEditionById(editions.stream()
				.filter(e -> "ADS NOW".equals(e.getName())).findFirst().get().getId()).block();

		oop20_21ModuleA = mApi.getModuleById(oop20_21.getModules().get(0).getId()).block();

		ads20_21ModuleA = mApi.getModuleById(ads20_21.getModules().get(0).getId()).block();
		ads20_21ModuleI = mApi.getModuleById(ads20_21.getModules().get(1).getId()).block();

		oopStudents = rds.students(eApi.getEditionParticipants(oop20_21.getId()).collectList().block());
		adsStudents = rds.students(eApi.getEditionParticipants(ads20_21.getId()).collectList().block());
	}

	private void initLabs() {
		oopL1 = ls.createLab(RegularLabCreateDTO.builder()
				.name("OOP Lab 1")
				.modules(Set.of(oop20_21ModuleA.getId()))
				.requestTypes(oop20_21ModuleA.getAssignments().stream()
						.collect(Collectors.toMap(AssignmentSummaryDTO::getId, a -> Set.of(QUESTION))))
				.rooms(rooms.stream()
						.map(RoomSummaryDTO::getId).collect(Collectors.toSet()))
				.communicationMethod(CommunicationMethod.TA_VISIT_STUDENT)
				.slot(new Slot(LocalDateTime.now(), LocalDateTime.now().plusMinutes(225)))
				.build(), oop20_21.getId(), LabService.LabType.REGULAR);

		oopL2 = ls.createLab(RegularLabCreateDTO.builder()
				.name("OOP Lab 2")
				.modules(Set.of(oop20_21ModuleA.getId()))
				.requestTypes(oop20_21ModuleA.getAssignments().stream()
						.collect(Collectors.toMap(AssignmentSummaryDTO::getId,
								a -> Set.of(QUESTION, SUBMISSION))))
				.rooms(rooms.stream()
						.filter(r -> !r.getName().contains("a"))
						.map(RoomSummaryDTO::getId).collect(Collectors.toSet()))
				.communicationMethod(CommunicationMethod.TA_VISIT_STUDENT)
				.slot(new Slot(LocalDateTime.now().plusWeeks(1),
						LocalDateTime.now().plusWeeks(1).plusMinutes(225)))
				.build(), oop20_21.getId(), LabService.LabType.REGULAR);

		oopE = ls.createLab(ExamLabCreateDTO.builder()
				.name("Exam")
				.slottedLabConfig(SlottedLabConfig.builder()
						.capacity(3)
						.duration(15L)
						.selectionOpensAt(LocalDateTime.now()).build())
				.examLabConfig(ExamLabConfig.builder()
						.percentage(15).build())
				.modules(Set.of(oop20_21ModuleA.getId()))
				.requestTypes(Map.of(oop20_21ModuleA.getAssignments().get(0).getId(), Set.of(SUBMISSION)))
				.rooms(Set.of(rooms.get(0).getId()))
				.communicationMethod(CommunicationMethod.TA_VISIT_STUDENT)
				.slot(new Slot(LocalDateTime.now(),
						LocalDateTime.now().plusMinutes(225)))
				.build(), oop20_21.getId(), LabService.LabType.REGULAR);

		adsL1I = ls.createLab(RegularLabCreateDTO.builder()
				.name("ADS Lab 1")
				.modules(Set.of(ads20_21ModuleI.getId()))
				.requestTypes(ads20_21ModuleI.getAssignments().stream()
						.collect(Collectors.toMap(AssignmentSummaryDTO::getId,
								a -> Set.of(QUESTION, SUBMISSION))))
				.rooms(rooms.stream()
						.map(RoomSummaryDTO::getId).collect(Collectors.toSet()))
				.communicationMethod(CommunicationMethod.TA_VISIT_STUDENT)
				.slot(new Slot(LocalDateTime.now(), LocalDateTime.now().plusMinutes(225)))
				.build(), ads20_21.getId(), LabService.LabType.REGULAR);

		adsL1A = ls.createLab(RegularLabCreateDTO.builder()
				.name("Analysis Lab 1")
				.modules(Set.of(ads20_21ModuleA.getId()))
				.requestTypes(ads20_21ModuleA.getAssignments().stream()
						.collect(Collectors.toMap(AssignmentSummaryDTO::getId, a -> Set.of(QUESTION))))
				.rooms(rooms.stream()
						.map(RoomSummaryDTO::getId).collect(Collectors.toSet()))
				.communicationMethod(CommunicationMethod.TA_VISIT_STUDENT)
				.slot(new Slot(LocalDateTime.now(), LocalDateTime.now().plusMinutes(225)))
				.build(), ads20_21.getId(), LabService.LabType.REGULAR);

		adsL2I = ls.createLab(RegularLabCreateDTO.builder()
				.name("ADS Lab 2")
				.modules(Set.of(ads20_21ModuleI.getId()))
				.requestTypes(ads20_21ModuleI.getAssignments().stream()
						.collect(Collectors.toMap(AssignmentSummaryDTO::getId,
								a -> Set.of(QUESTION, SUBMISSION))))
				.rooms(rooms.stream()
						.filter(r -> !r.getName().contains("A"))
						.map(RoomSummaryDTO::getId).collect(Collectors.toSet()))
				.communicationMethod(CommunicationMethod.TA_VISIT_STUDENT)
				.slot(new Slot(LocalDateTime.now().plusWeeks(1),
						LocalDateTime.now().plusWeeks(1).plusMinutes(225)))
				.build(), ads20_21.getId(), LabService.LabType.REGULAR);
	}

	private void initRequests() {
		oopStudents.forEach(s -> {
			rs.createRequest(new DevRequestCreateDTO(
					QUESTION,
					"What should I do here?",
					"Ah well",
					null,
					oop20_21ModuleA.getAssignments().get(0).getId(),
					rooms.get(0).getId(),
					oopL1), s.getId(), false);

			rs.createRequest(new DevRequestCreateDTO(
					QUESTION,
					"What should I do here?",
					"Ah well",
					null,
					oop20_21ModuleA.getAssignments().get(0).getId(),
					rooms.get(0).getId(),
					oopL2), s.getId(), false);
		});
	}

	/**
	 * Class overriding the default validation of the request create DTO as default validation requires a
	 * request scope to be active. The caches for sessions are managed per-request.
	 */
	private static class DevRequestCreateDTO extends RequestCreateDTO {
		public DevRequestCreateDTO(
				@NotNull RequestType requestType,
				@Size(max = 250) String comment,
				@Size(max = 500) String question,
				TimeSlotIdDTO timeSlot,
				@NotNull Long assignment,
				@NotNull Long room, Lab lab) {
			super(requestType, comment, question, timeSlot, assignment, room, lab);
		}

		@Override
		public void validate() {
		}
	}
}
