/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.repository.LabRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;

@Controller
@PreAuthorize("@permissionService.isAdmin()")
@RequestMapping("/admin")
public class AdminController {

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private EditionControllerApi eApi;

	@Autowired
	private EditionCacheManager eCache;

	/**
	 * Adds a model attribute to every template resolution for recognizing the type of page we are in. For
	 * this controller all pages will be in the "admin"-area.
	 *
	 * @return The generic page name: "admin".
	 */
	@ModelAttribute("page")
	public static String page() {
		return "admin";
	}

	@GetMapping
	public String getAdminPage(Model model) {
		Period currPeriod = new Period().start(LocalDateTime.now().minusDays(1))
				.end(LocalDateTime.now().plusDays(7));
		var editions = eCache.get(Objects.requireNonNull(eApi.getAllEditionsActiveDuringPeriod(currPeriod)
				.map(EditionSummaryDTO::getId).collectList().block()));
		var sessions = editions.stream().map(EditionDetailsDTO::getSessions).flatMap(List::stream)
				.collect(Collectors.toList());
		model.addAttribute("sessions", sessions);
		long count;
		if ((count = onGoingSession(sessions)) > 0) {
			model.addAttribute("warningMessage", String.format("Currently %d sessions " +
					"ongoing!", count));
		}
		return "admin/view";
	}

	private long onGoingSession(List<SessionSummaryDTO> sessions) {
		LocalDateTime now = LocalDateTime.now();
		return sessions.stream().filter(s -> s.getStart().isBefore(now) && s.getEnd().isAfter(now)).count();
	}

}
