/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static java.time.LocalDateTime.now;
import static nl.tudelft.labracore.lib.LabracoreApiUtil.fromPageable;
import static nl.tudelft.queue.PageUtil.toPage;

import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.RoleControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.labracore.lib.security.user.AuthenticatedPerson;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.cache.*;
import nl.tudelft.queue.dto.create.QueueRoleCreateDTO;
import nl.tudelft.queue.dto.view.LabSummaryDTO;
import nl.tudelft.queue.repository.LabRepository;
import nl.tudelft.queue.service.EditionService;
import nl.tudelft.queue.service.RoleDTOService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PageableDefault;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

@Controller
public class EditionController {

	@Autowired
	private LabRepository lr;

	@Autowired
	private EditionControllerApi eApi;

	@Autowired
	private PersonControllerApi pApi;

	@Autowired
	private RoleControllerApi rApi;

	@Autowired
	private RoleDTOService roleDTOService;

	@Autowired
	private AssignmentCacheManager aCache;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private ModuleCacheManager mCache;

	@Autowired
	private RoleCacheManager rCache;

	@Autowired
	private RoomCacheManager rmCache;

	@Autowired
	private SessionCacheManager sCache;

	@Autowired
	private StudentGroupCacheManager sgCache;

	@Autowired
	private EditionService es;

	/**
	 * Sets a model attribute statically within every Thymeleaf resolution. This model attribute is to
	 * indicate the main tab that course pages are in.
	 *
	 * @return The name of the tab that these pages are on ("courses").
	 */
	@ModelAttribute("page")
	public static String page() {
		return "editions";
	}

	/**
	 * Gets the page listing all editions. This page incorporates pageables to index many different editions
	 * and keep the overview small.
	 *
	 * @param  person   The currently authenticated person asking for the editions page.
	 * @param  pageable The pageable object representing the current page and size to display.
	 * @param  model    The model to fill out for Thymeleaf template resolution.
	 * @return          The Thymeleaf template to resolve.
	 */
	@GetMapping("/editions")
	public String getCourseList(@AuthenticatedPerson Person person,
			@PageableDefault(sort = "id", direction = Sort.Direction.DESC) Pageable pageable,
			Model model) {
		var editions = eApi
				.getEditionsPageActiveOrTaughtBy(person.getId(), fromPageable(pageable))
				.block();
		eCache.register(editions.getContent());

		model.addAttribute("editions",
				new PageImpl<>(editions.getContent(), pageable, editions.getTotalElements()));

		return "edition/index";
	}

	/**
	 * Gets the confirmation page for users that have the intention of enrolling in the given course edition.
	 * Entrance into this page is given whenever the authenticated person does not have a role in the given
	 * edition yet.
	 *
	 * @param  editionId The id of the edition the user wants to enrol in.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/enrol")
	@PreAuthorize("@permissionService.canEnrolForEdition(#editionId)")
	public String getEditionEnrolView(@PathVariable Long editionId, Model model) {
		model.addAttribute("edition", eCache.getOrThrow(editionId));

		return "edition/enrol";
	}

	/**
	 * Posts an enrolment request for one person in one edition. This method handles the post by calling the
	 * appropriate enrolment function at Labracore.
	 *
	 * @param  person    The currently authenticated person that is trying to enrol for a course edition.
	 * @param  editionId The id of the edition the user wants to enrol in.
	 * @return           A redirect to the edition info page for the given edition edition.
	 */
	@PostMapping("/edition/{editionId}/enrol")
	@PreAuthorize("@permissionService.canEnrolForEdition(#editionId)")
	public String enrolForEdition(@AuthenticatedPerson Person person,
			@PathVariable Long editionId) {
		eApi.addStudentsToEdition(editionId, List.of(person.getUsername())).block();

		return "redirect:/edition/" + editionId;
	}

	/**
	 * Gets the main course edition info panel page. This page shows the most generic information on a course
	 * edition taken from Labracore.
	 *
	 * @param  editionId The id of the edition to display.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}")
	public String getEditionView(@PathVariable Long editionId, Model model) {
		model.addAttribute("edition", eCache.getOrThrow(editionId));

		return "edition/view/info";
	}

	/**
	 * Gets the course edition info panel on participants. This page should be shown to people that have
	 * permission to edit the participants list and can see all students.
	 *
	 * @param  editionId The id of the edition for which to display participants.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/participants")
	@PreAuthorize("@permissionService.canManageParticipants(#editionId)")
	public String getEditionParticipantsView(@PathVariable Long editionId, Model model,
			@RequestParam(value = "student-search", required = false) String studentSearch,
			@PageableDefault(size = 25) Pageable pageable) {
		var edition = eCache.getOrThrow(editionId);
		var students = roleDTOService.students(edition);

		if (studentSearch != null) {
			students = es.studentsMatchingFilter(students, studentSearch);
		}

		model.addAttribute("edition", edition);
		model.addAttribute("students", toPage(pageable, students));

		return "edition/view/participants";
	}

	/**
	 * Gets the course edition info panel on modules. This page should be shown to all students and
	 * assistants. This page only displays information on the modules, assignments and groups as can be found
	 * in Labracore.
	 *
	 * @param  editionId The id of the edition for which to display the modules.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/modules")
	@PreAuthorize("@permissionService.canViewEdition(#editionId)")
	public String getEditionModulesView(@PathVariable Long editionId, Model model) {
		var edition = eCache.getOrThrow(editionId);
		var modules = mCache.get(edition.getModules().stream().map(ModuleSummaryDTO::getId));
		sgCache.get(modules.stream()
				.flatMap(m -> m.getGroups().stream().map(StudentGroupSmallSummaryDTO::getId)));

		model.addAttribute("edition", edition);
		model.addAttribute("modules", modules);
		model.addAttribute("groups", modules.stream()
				.collect(Collectors.toMap(ModuleDetailsDTO::getId,
						m -> sgCache.get(m.getGroups().stream().map(StudentGroupSmallSummaryDTO::getId)))));

		return "edition/view/modules";
	}

	/**
	 * Gets the course edition info panel on labs. This page should shown to all those that need to interact
	 * with labs. This page shows all labs currently available and has links to view all labs and a button to
	 * create a new lab.
	 *
	 * @param  editionId The id of the edition for which to display labs.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/labs")
	@PreAuthorize("@permissionService.canViewEdition(#editionId)")
	public String getEditionLabsView(@PathVariable Long editionId, Model model) {
		var edition = eCache.getOrThrow(editionId);

		// Make sure that session details are cached and then convert to summaries.
		var sessions = sCache.get(edition.getSessions().stream()
				.map(SessionSummaryDTO::getId));
		var labs = getLabSummariesFromSessions(sessions, s -> true);

		model.addAttribute("edition", edition);
		model.addAttribute("labs", labs);

		return "edition/view/labs";
	}

	/**
	 * Gets the participant add page. This page is used to add new participants to an edition.
	 *
	 * @param  editionId The id of the edition to which a participant should be added.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/participants/create")
	@PreAuthorize("@permissionService.canManageParticipants(#editionId)")
	public String getAddParticipantPage(@PathVariable Long editionId, Model model) {
		model.addAttribute("edition", eCache.getOrThrow(editionId));
		model.addAttribute("role", new QueueRoleCreateDTO(editionId));

		return "edition/create/participant";
	}

	/**
	 * Posts a single participant addition to the server. This action comes from the participant add page.
	 *
	 * @param  editionId The id of the edition to which a participant should be added.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@PostMapping("/edition/{editionId}/participants/create")
	@PreAuthorize("@permissionService.canManageParticipants(#editionId)")
	public String createParticipant(@PathVariable Long editionId, Model model,
			QueueRoleCreateDTO dto) {
		RoleCreateDTO create = dto.apply();
		if (dto.hasErrors()) {
			model.addAttribute("edition", eCache.getOrThrow(editionId));
			model.addAttribute("role", dto);

			return "edition/create/participant";
		}

		pApi.getPersonByUsername(dto.getUsername())
				.flatMap(person -> rApi.addRole(create.person(new PersonIdDTO().id(person.getId()))))
				.block();

		return "redirect:/edition/" + editionId + "/participants";
	}

	/**
	 * Gets the participant remove page. This page is used to remove a specific participant from the given
	 * edition.
	 *
	 * @param  editionId The id of the edition from which the participant should be removed.
	 * @param  personId  The id of the person who should be removed.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/participants/{personId}/remove")
	@PreAuthorize("@permissionService.canManageParticipants(#editionId)")
	public String getRemoveParticipantPage(@PathVariable Long editionId, @PathVariable Long personId,
			Model model) {
		EditionDetailsDTO edition = eCache.getOrThrow(editionId);

		model.addAttribute("edition", edition);
		model.addAttribute("role", rCache.getOrThrow(new Id()
				.personId(personId).editionId(editionId)));

		return "edition/remove/participant";
	}

	/**
	 * Removes a participant with the given personId from the given edition.
	 *
	 * @param  editionId The id of the edition from which the participant should be removed.
	 * @param  personId  The id of the person who should be removed.
	 * @return           A redirect to the edition participants overview.
	 */
	@PostMapping("/edition/{editionId}/participants/{personId}/remove")
	@PreAuthorize("@permissionService.canManageParticipants(#editionId)")
	public String removeParticipant(@PathVariable Long editionId, @PathVariable Long personId) {
		eApi.removePersonFromEdition(editionId, personId).block();

		return "redirect:/edition/" + editionId + "/participants";
	}

	/**
	 * Blocks a person with the given id from the given edition.
	 *
	 * @param  editionId The id of the edition to block someone from.
	 * @param  personId  The id of the person to block.
	 * @return           A redirect to the edition participants overview.
	 */
	@PostMapping("/edition/{editionId}/participants/{personId}/block")
	@PreAuthorize("@permissionService.canManageParticipants(#editionId)")
	public String blockParticipant(@PathVariable Long editionId, @PathVariable Long personId) {
		eApi.blockPersonInEdition(editionId, personId).block();

		return "redirect:/edition/" + editionId + "/participants";
	}

	/**
	 * Gets the page to confirm for a student that they want to unenrol from the given course edition.
	 *
	 * @param  editionId The id of the edition from which a student wants to unenrol.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/leave")
	@PreAuthorize("@permissionService.canViewEdition(#editionId)")
	public String getParticipantLeavePage(@PathVariable Long editionId,
			Model model) {
		model.addAttribute("edition", eCache.getOrThrow(editionId));

		return "edition/view/leave";
	}

	/**
	 * Processes a POST request from the user wanting to leave the given edition. Instead of deleting the role
	 * of the user from the database entirely, we opt to block their role and thus disallow them access to the
	 * edition.
	 *
	 * @param  user      The currently authenticated user as a Person object.
	 * @param  editionId The id of the edition that the student wants to leave.
	 * @return           A redirect back to the edition view page.
	 */
	@PostMapping("/edition/{editionId}/leave")
	@PreAuthorize("@permissionService.canViewEdition(#editionId)")
	public String participantLeave(@AuthenticatedPerson Person user,
			@PathVariable Long editionId) {
		eApi.blockPersonInEdition(editionId, user.getId()).block();

		return "redirect:/edition/" + editionId;
	}

	/**
	 * Gets the edition status page. This page is a status overview for the entire edition.
	 *
	 * @param  editionId The id of the edition for which to view the edition status.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/status")
	@PreAuthorize("@permissionService.canViewEditionStatus(#editionId)")
	public String status(@PathVariable Long editionId,
			Model model) {
		var edition = eCache.getOrThrow(editionId);
		var sessions = sCache.get(edition.getSessions().stream().map(SessionSummaryDTO::getId));

		model.addAttribute("edition", edition);

		model.addAttribute("activeLabs", getLabSummariesFromSessions(
				sessions, s -> s.getStart().isBefore(now()) && now().isBefore(s.getEnd())));
		model.addAttribute("inactiveLabs", getLabSummariesFromSessions(
				sessions, s -> !(s.getStart().isBefore(now()) && now().isBefore(s.getEnd()))));

		model.addAttribute("assignments", aCache.get(sessions.stream()
				.flatMap(s -> s.getAssignments().stream().map(AssignmentSummaryDTO::getId)).distinct()));
		model.addAttribute("rooms", rmCache.get(sessions.stream()
				.flatMap(s -> s.getRooms().stream().map(RoomSummaryDTO::getId)).distinct()));

		return "edition/view/status";
	}

	/**
	 * Gets lab views for the given list of sessions after applying the given predicate.
	 *
	 * @param  sessions The sessions that are to be displayed through lab view DTOs.
	 * @param  p        The predicate to apply to the sessions to filter out the relevant labs.
	 * @return          The list of LabViewDTOs representing the selected sessions (and labs).
	 */
	private List<LabSummaryDTO> getLabSummariesFromSessions(List<SessionDetailsDTO> sessions,
			Predicate<SessionDetailsDTO> p) {
		return View.convert(lr.findAllBySessions(
				sessions.stream().filter(p)
						.map(SessionDetailsDTO::getId)
						.collect(Collectors.toList())),
				LabSummaryDTO.class);
	}
}
