/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static nl.tudelft.labracore.api.dto.PersonDetailsDTO.DefaultRoleEnum.ADMIN;

import java.time.LocalDateTime;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.RoleControllerApi;
import nl.tudelft.labracore.api.dto.EditionDetailsDTO;
import nl.tudelft.labracore.api.dto.EditionSummaryDTO;
import nl.tudelft.labracore.api.dto.PersonDetailsDTO;
import nl.tudelft.labracore.lib.security.user.AuthenticatedPerson;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.dto.view.FeedbackViewDTO;
import nl.tudelft.queue.model.Feedback;
import nl.tudelft.queue.repository.FeedbackRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;

@Controller
public class HomeController {

	@Autowired
	private FeedbackRepository fr;

	@Autowired
	private EditionControllerApi eApi;

	@Autowired
	private PersonControllerApi pApi;

	@Autowired
	private RoleControllerApi rApi;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private PersonCacheManager pCache;

	/**
	 * Sets a model attribute statically within every Thymeleaf resolution. This model attribute is to
	 * indicate the main tab that Home pages are in.
	 *
	 * @return The name of the tab that these pages are on ("main").
	 */
	@ModelAttribute("page")
	public String page() {
		return "main";
	}

	/**
	 * @return The path for the privacy statement.
	 */
	@GetMapping("/privacy")
	public String privacyStatement() {
		return "home/privacy";
	}

	/**
	 * @return The path for the about page.
	 */
	@GetMapping("/about")
	public String about() {
		return "home/about";
	}

	/**
	 * Maps the index/root url to a page. When the user is logged in correctly with an
	 * AnonymousAuthenticationToken, the user is directed to a dashboard view of the Queue. When the user is
	 * not correctly logged in, they are redirected to the introductory page of the Queue.
	 *
	 * @param  person The person currently authenticated for which a dashboard might be shown.
	 * @param  model  The model to be filled out when the user has editions and/or labs to show on their
	 *                dashboard.
	 * @return        The path to the corresponding Thymeleaf template.
	 */
	@GetMapping("/")
	public String index(@AuthenticatedPerson(required = false) Person person, Model model) {
		if (person != null) {
			PersonDetailsDTO pd = pApi.getPersonById(person.getId()).block();

			var roles = rApi
					.getPersonRolesInActiveEditions(Objects.requireNonNull(pd).getId())
					.collectList().block();

			var editions = eCache.get(roles.stream().map(r -> r.getEdition().getId()))
					.stream().collect(Collectors.toMap(EditionDetailsDTO::getId, Function.identity()));

			if (pd.getDefaultRole() == ADMIN) {
				var runningEditions = eCache.get(Objects.requireNonNull(
						eApi.getAllEditionsActiveAtDate(LocalDateTime.now())
								.map(EditionSummaryDTO::getId).collectList().block()));

				model.addAttribute("runningEditions", runningEditions);
			}

			model.addAttribute("editions", editions);
			model.addAttribute("roles", roles);
			model.addAttribute("user", pd);

			return "home/dashboard";
		}
		return "home/index";
	}

	/**
	 * Maps the own feedback url to a page. The Feedback page displays feedback for the currently
	 * authenticated user.
	 *
	 * @param  person   The person that is currently authenticated.
	 * @param  model    The model to be filled out for Thymeleaf resolution.
	 * @param  pageable The Pageable object representing the current page state.
	 * @return          The feedback page location to be filled in by Thymeleaf.
	 */
	@GetMapping("/feedback")
	@PreAuthorize("@permissionService.canViewOwnFeedback()")
	public String ownFeedback(@AuthenticatedPerson Person person,
			Model model, Pageable pageable) {
		fillInFeedbackModel(person.getId(), model, pageable);

		return "home/feedback";
	}

	/**
	 * Maps the feedback url to a page. The Feedback page displays feedback for a user with the given ID.
	 *
	 * @param  id       The id of the person for which feedback will be shown.
	 * @param  model    The model to be filled out for Thymeleaf resolution.
	 * @param  pageable The Pageable object representing the current page state.
	 * @return          The feedback page location to be filled in by Thymeleaf.
	 */
	@GetMapping("/feedback/{id}")
	@PreAuthorize("@permissionService.canViewFeedback(#id)")
	public String feedback(@PathVariable("id") Long id,
			Model model, Pageable pageable) {
		fillInFeedbackModel(id, model, pageable);

		return "home/feedback";
	}

	private void fillInFeedbackModel(Long assistantId, Model model, Pageable pageable) {
		var assistant = pCache.getOrThrow(assistantId);

		Page<Feedback> feedback = fr.findByAssistant(assistantId, pageable);

		model.addAttribute("assistant", assistant);
		model.addAttribute("feedback",
				feedback.map(fb -> View.convert(fb, FeedbackViewDTO.class)));
	}

}
