/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.transaction.Transactional;

import nl.tudelft.labracore.api.dto.AssignmentSummaryDTO;
import nl.tudelft.labracore.api.dto.EditionSummaryDTO;
import nl.tudelft.labracore.api.dto.ModuleSummaryDTO;
import nl.tudelft.labracore.lib.security.user.AuthenticatedPerson;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.librador.resolver.annotations.PathEntity;
import nl.tudelft.queue.cache.*;
import nl.tudelft.queue.dto.create.RequestCreateDTO;
import nl.tudelft.queue.dto.create.labs.ExamLabCreateDTO;
import nl.tudelft.queue.dto.create.labs.RegularLabCreateDTO;
import nl.tudelft.queue.dto.create.labs.SlottedLabCreateDTO;
import nl.tudelft.queue.dto.patch.RequestPatchDTO;
import nl.tudelft.queue.dto.patch.labs.ExamLabPatchDTO;
import nl.tudelft.queue.dto.patch.labs.RegularLabPatchDTO;
import nl.tudelft.queue.dto.patch.labs.SlottedLabPatchDTO;
import nl.tudelft.queue.dto.view.RequestViewDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.enums.LabType;
import nl.tudelft.queue.model.labs.ExamLab;
import nl.tudelft.queue.model.labs.RegularLab;
import nl.tudelft.queue.model.labs.SlottedLab;
import nl.tudelft.queue.repository.LabRepository;
import nl.tudelft.queue.service.LabService;
import nl.tudelft.queue.service.PermissionService;
import nl.tudelft.queue.service.RequestService;
import nl.tudelft.queue.service.RequestTableService;

import org.modelmapper.ModelMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

@Controller
public class LabController {
	@Autowired
	private LabRepository labRepository;

	@Autowired
	private LabService ls;

	@Autowired
	private RequestTableService rts;

	@Autowired
	private PermissionService ps;

	@Autowired
	private RequestService rs;

	@Autowired
	private AssignmentCacheManager aCache;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private EditionCollectionCacheManager ecCache;

	@Autowired
	private ModuleCacheManager mCache;

	@Autowired
	private SessionCacheManager sCache;

	@Autowired
	private RoomCacheManager rCache;

	/**
	 * Gets the page with information on the lab with the given id. This page is different for TAs than for
	 * students. For students, this page shows their history of requests and their current requests. For TAs,
	 * this page mostly shows the full history of all requests that have previously happened in a lab. For
	 * finished labs, this is the main info available on the lab page, together with some general information
	 * on the lab.
	 *
	 * @param  lab    The lab to display on the page.
	 * @param  person The currently authenticated person.
	 * @param  model  The model to fill out for Thymeleaf template resolution.
	 * @return        The Thymeleaf template to resolve.
	 */
	@GetMapping("/lab/{lab}")
	@PreAuthorize("@permissionService.canViewLab(#lab)")
	public String getLabView(@PathEntity Lab lab,
			@AuthenticatedPerson Person person,
			Model model) {
		setEnqueuePageAttributes(lab, model);

		// Either get all requests for the lab if the person is an assistant or just those specific to the person.
		model.addAttribute("requests",
				rts.convertRequestsToView(ps.canTakeRequest(lab.getId()) ? lab.getRequests()
						: lab.getAllRequestsForPerson(person.getId()))
						.stream()
						.sorted(Comparator.comparing(RequestViewDTO::getCreatedAt).reversed())
						.collect(Collectors.toList()));

		lab.getOpenRequestForPerson(person.getId()).ifPresent(r -> {
			model.addAttribute("current", View.convert(r, RequestViewDTO.class));
			model.addAttribute("currentDto", new ModelMapper().map(r, RequestPatchDTO.class));
		});

		model.addAttribute("modules", mCache.get(lab.getModules().stream()));

		return "lab/view/" + lab.getType().name().toLowerCase();
	}

	/**
	 * Gets the student enqueue view. This page displays the form that needs to be filled out by the student
	 * to successfully enrol into the given lab.
	 *
	 * @param  lab   The lab the student is enrolling in.
	 * @param  model The model to fill out for Thymeleaf template resolution.
	 * @return       The Thymeleaf template to resolve.
	 */
	@GetMapping("/lab/{lab}/enqueue")
	@PreAuthorize("@permissionService.canEnqueueSelf(#lab.id)")
	public String getEnqueueView(@PathEntity Lab lab,
			Model model) {
		model.addAttribute("request", new RequestCreateDTO());
		setEnqueuePageAttributes(lab, model);

		return "lab/enqueue";
	}

	/**
	 * Performs the enqueue action. This action allows a student to create a new request in a lab they are
	 * participating in.
	 *
	 * @param  student The student making the request.
	 * @param  lab     The lab in which the request is to be created.
	 * @param  dto     The dto representing the values the student has entered for creating their request.
	 * @return         The Thymeleaf template to resolve or a redirect back to the lab page.
	 */
	@PostMapping("/lab/{lab}/enqueue")
	@PreAuthorize("@permissionService.canEnqueueSelf(#lab.id)")
	public String enqueueInLab(@AuthenticatedPerson Person student,
			@PathEntity Lab lab,
			RequestCreateDTO dto) {
		dto.setLab(lab);

		rs.createRequest(dto, student.getId(), true);

		return "redirect:/lab/" + lab.getId();
	}

	/**
	 * Performs the 'revoke request' command and redirects the user back to the lab overview page.
	 *
	 * @param  student The student that is currently authenticated and is requesting a revoke.
	 * @param  lab     The lab the student is revoking their request in.
	 * @return         A redirect to the lab overview page.
	 */
	@PostMapping("/lab/{lab}/revoke")
	@PreAuthorize("@permissionService.canRevokeFromLab(#lab.id)")
	public String revokeRequest(@AuthenticatedPerson Person student,
			@PathEntity Lab lab) {
		lab.getOpenRequestForPerson(student.getId())
				.filter(r -> r.getEventInfo().getStatus().isPending())
				.ifPresent(rs::revokeRequest);

		return "redirect:/lab/" + lab.getId();
	}

	/**
	 * Gets the lab creation page. This page should be viewable by all teachers and managers (those that are
	 * allowed to create labs). This page allows for editing a lab before final creation of that lab.
	 *
	 * @param  editionId The id of the edition to create the lab in.
	 * @param  model     The model to fill out for Thymeleaf template resolution.
	 * @return           The Thymeleaf template to resolve.
	 */
	@GetMapping("/edition/{editionId}/lab/create")
	@PreAuthorize("@permissionService.canManageLabs(#editionId)")
	public String getLabCreateView(@PathVariable Long editionId,
			@RequestParam LabType type, Model model) {
		model.addAttribute("dto", type.newCreateDto());
		model.addAttribute("edition", eCache.getOrThrow(editionId));
		model.addAttribute("lType", type);

		setLabEditingPageAttributes(List.of(editionId), model);

		return "lab/create/" + type.name().toLowerCase();
	}

	/**
	 * Gets the lab creation page. This page should be viewable by all teachers and managers (those that are
	 * allowed to create labs). This page allows for editing a lab before final creation of that lab.
	 *
	 * @param  editionCollectionId The id of the edition collection to create lab in.
	 * @param  model               The model to fill out for Thymeleaf template resolution.
	 * @return                     The Thymeleaf template to resolve.
	 */
	@GetMapping("/shared-edition/{editionCollectionId}/lab/create")
	@PreAuthorize("@permissionService.canManageLabs(#editionCollectionId)")
	public String getSharedLabCreateView(@PathVariable Long editionCollectionId,
			@RequestParam LabType type, Model model) {
		var editionCollection = ecCache.getOrThrow(editionCollectionId);

		model.addAttribute("dto", type.newCreateDto());
		model.addAttribute("ec", editionCollection);

		setLabEditingPageAttributes(editionCollection.getEditions().stream()
				.map(EditionSummaryDTO::getId).collect(Collectors.toList()), model);

		return "lab/create/" + type.name().toLowerCase();
	}

	/**
	 * Creates a new lab using the lab create DTO provided through the body of the POST request. Creates a lab
	 * of the regular type.
	 *
	 * @param  editionId The id of the edition this lab is part of.
	 * @param  dto       The dto to apply to create a new lab.
	 * @return           A redirect to the created lab page.
	 */
	@PostMapping("/edition/{editionId}/lab/create/regular")
	@PreAuthorize("@permissionService.canManageLabs(#editionId)")
	public String createRegularLab(@PathVariable Long editionId,
			RegularLabCreateDTO dto) {
		return "redirect:/lab/" + ls.createLab(dto, editionId, LabService.LabType.REGULAR).getId();
	}

	/**
	 * Creates a new lab using the lab create DTO provided through the body of the POST request. Creates a lab
	 * of the slotted type.
	 *
	 * @param  editionId The id of the edition this lab is part of.
	 * @param  dto       The dto to apply to create a new lab.
	 * @return           A redirect to the created lab page.
	 */
	@PostMapping("/edition/{editionId}/lab/create/slotted")
	@PreAuthorize("@permissionService.canManageLabs(#editionId)")
	public String createSlottedLab(@PathVariable Long editionId,
			SlottedLabCreateDTO dto) {
		return "redirect:/lab/" + ls.createLab(dto, editionId, LabService.LabType.REGULAR).getId();
	}

	/**
	 * Creates a new lab using the lab create DTO provided through the body of the POST request. Creates a lab
	 * of the exam type.
	 *
	 * @param  editionId The id of the edition this lab is part of.
	 * @param  dto       The dto to apply to create a new lab.
	 * @return           A redirect to the created lab page.
	 */
	@PostMapping("/edition/{editionId}/lab/create/exam")
	@PreAuthorize("@permissionService.canManageLabs(#editionId)")
	public String createExamLab(@PathVariable Long editionId,
			ExamLabCreateDTO dto) {
		return "redirect:/lab/" + ls.createLab(dto, editionId, LabService.LabType.REGULAR).getId();
	}

	/**
	 * Creates a new lab using the lab create DTO provided through the body of the POST request. This method
	 * creates one specifically for a shared edition. Creates a lab of the regular type.
	 *
	 * @param  editionCollectionId The id of the edition collection the lab will be part of.
	 * @param  dto                 The dto to apply to create a new lab.
	 * @return                     A redirect to the created lab page.
	 */
	@PostMapping("/shared-edition/{editionCollectionId}/lab/create/regular")
	@PreAuthorize("@permissionService.canManageLabs(#editionCollectionId)")
	public String createRegularSharedLab(@PathVariable Long editionCollectionId,
			RegularLabCreateDTO dto) {
		return "redirect:/lab/" + ls.createLab(dto, editionCollectionId, LabService.LabType.SHARED).getId();
	}

	/**
	 * Creates a new lab using the lab create DTO provided through the body of the POST request. This method
	 * creates one specifically for a shared edition. Creates a lab of the slotted type.
	 *
	 * @param  editionCollectionId The id of the edition collection the lab will be part of.
	 * @param  dto                 The dto to apply to create a new lab.
	 * @return                     A redirect to the created lab page.
	 */
	@PostMapping("/shared-edition/{editionCollectionId}/lab/create/slotted")
	@PreAuthorize("@permissionService.canManageLabs(#editionCollectionId)")
	public String createSlottedSharedLab(@PathVariable Long editionCollectionId,
			SlottedLabCreateDTO dto) {
		return "redirect:/lab/" + ls.createLab(dto, editionCollectionId, LabService.LabType.SHARED).getId();
	}

	/**
	 * Creates a new lab using the lab create DTO provided through the body of the POST request. This method
	 * creates one specifically for a shared edition. Creates a lab of the exam type.
	 *
	 * @param  editionCollectionId The id of the edition collection the lab will be part of.
	 * @param  dto                 The dto to apply to create a new lab.
	 * @return                     A redirect to the created lab page.
	 */
	@PostMapping("/shared-edition/{editionCollectionId}/lab/create/exam")
	@PreAuthorize("@permissionService.canManageLabs(#editionCollectionId)")
	public String createExamSharedLab(@PathVariable Long editionCollectionId,
			ExamLabCreateDTO dto) {
		return "redirect:/lab/" + ls.createLab(dto, editionCollectionId, LabService.LabType.SHARED).getId();
	}

	/**
	 * Gets the confirmation view for deleting a single lab.
	 *
	 * @param  lab   The lab to delete.
	 * @param  model The model to fill out for Thymeleaf template resolution.
	 * @return       The Thymeleaf template to resolve.
	 */
	@GetMapping("/lab/{lab}/delete")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String getLabDeleteView(@PathEntity Lab lab,
			Model model) {
		ls.setOrganizationInModel(lab, model);

		return "/lab/remove";
	}

	/**
	 * Handles a POST request to the lab delete endpoint by setting the deleted at date for the given lab.
	 *
	 * @param  lab The lab to delete.
	 * @return     A redirect back to the edition lab overview.
	 */
	@PostMapping("/lab/{lab}/delete")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String deleteLab(@PathEntity Lab lab) {
		ls.deleteLab(lab);
		var session = sCache.getOrThrow(lab.getSession());

		// TODO: Redirect to an edition collection specific page for shared labs
		return "redirect:/edition/" + session.getEditions().get(0).getId() + "/labs";
	}

	/**
	 * Gets the lab creation page copied from the lab targeted in the url of this request.
	 *
	 * @param  lab   The lab that is targeted for copying.
	 * @param  model The model to fill out for Thymeleaf template resolution.
	 * @return       The Thymeleaf template to resolve.
	 */
	@GetMapping("/lab/{lab}/copy")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String getLabCopyView(@PathEntity Lab lab,
			Model model) {
		var session = sCache.getOrThrow(lab.getSession());

		model.addAttribute("dto", lab.copyLabCreateDTO(session));

		setLabEditingPageAttributes(lab, model);

		return "lab/create/" + lab.getType().name().toLowerCase();
	}

	/**
	 * Gets the lab editing page.
	 *
	 * @param  lab   The lab that is to be edited by this page.
	 * @param  model The model to fill out for Thymeleaf template resolution.
	 * @return       The Thymeleaf template to resolve.
	 */
	@GetMapping("/lab/{lab}/edit")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String getLabUpdateView(@PathEntity Lab lab, Model model) {
		model.addAttribute("dto", lab.newPatchDTO());
		model.addAttribute("lab", lab);

		setLabEditingPageAttributes(lab, model);

		return "lab/edit/" + lab.getType().name().toLowerCase();
	}

	/**
	 * Updates the given lab entity using the given lab patch DTO. The lab in question is of the regular type.
	 *
	 * @param  lab The lab to update.
	 * @param  dto The dto representing the changes to apply to the lab.
	 * @return     A redirect to the lab view page.
	 */
	@Transactional
	@PostMapping("/lab/{lab}/edit/regular")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String updateLab(@PathEntity RegularLab lab, RegularLabPatchDTO dto) {
		ls.updateLab(dto, lab);
		return "redirect:/lab/" + lab.getId();
	}

	/**
	 * Updates the given lab entity using the given lab patch DTO. The lab in question is of the slotted type.
	 *
	 * @param  lab The lab to update.
	 * @param  dto The dto representing the changes to apply to the lab.
	 * @return     A redirect to the lab view page.
	 */
	@Transactional
	@PostMapping("/lab/{lab}/edit/slotted")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String updateLab(@PathEntity SlottedLab lab, SlottedLabPatchDTO dto) {
		ls.updateLab(dto, lab);
		return "redirect:/lab/" + lab.getId();
	}

	/**
	 * Updates the given lab entity using the given lab patch DTO. The lab in question is of the exam type.
	 *
	 * @param  lab The lab to update.
	 * @param  dto The dto representing the changes to apply to the lab.
	 * @return     A redirect to the lab view page.
	 */
	@Transactional
	@PostMapping("/lab/{lab}/edit/exam")
	@PreAuthorize("@permissionService.canManageLab(#lab)")
	public String updateLab(@PathEntity ExamLab lab, ExamLabPatchDTO dto) {
		ls.updateLab(dto, lab);
		return "redirect:/lab/" + lab.getId();
	}

	/**
	 * Sets the model attributes for an enqueueing or editing requests page.
	 *
	 * @param lab   The lab which the request takes place in.
	 * @param model The model to fill using edition and lab information.
	 */
	private void setEnqueuePageAttributes(Lab lab, Model model) {
		var session = sCache.getOrThrow(lab.getSession());

		model.addAttribute("lab", lab.toViewDTO());
		ls.setOrganizationInModel(session, model);

		model.addAttribute("rooms", session.getRooms());
		model.addAttribute("assignments", aCache.get(lab.getAllowedRequests().stream()
				.map(AllowedRequest::getAssignment).collect(Collectors.toList())));
		model.addAttribute("types", lab.getAllowedRequests().stream()
				.collect(Collectors.groupingBy(AllowedRequest::getAssignment,
						Collectors.mapping(ar -> ar.getType().displayName(), Collectors.toSet()))));
	}

	/**
	 * Sets the model attributes for a lab editing or creation page.
	 *
	 * @param editionId The id of the edition the lab will be in.
	 * @param model     The model to fill using edition information.
	 */
	private void setLabEditingPageAttributes(List<Long> editionIds, Model model) {
		var editions = eCache.get(editionIds);

		var modules = mCache.get(editions.stream()
				.flatMap(e -> e.getModules().stream().map(ModuleSummaryDTO::getId).distinct()));
		var assignments = aCache.get(modules.stream()
				.flatMap(m -> m.getAssignments().stream().map(AssignmentSummaryDTO::getId)).distinct());

		model.addAttribute("editions", editions);

		model.addAttribute("modules", modules);
		model.addAttribute("assignments", assignments);

		model.addAttribute("rooms", rCache.getAll());
	}

	/**
	 * Sets the attributes necessary for the page for editing a lab.
	 *
	 * @param lab   The (existing) lab that is to be edited.
	 * @param model The model to fill using lab information.
	 */
	private void setLabEditingPageAttributes(Lab lab, Model model) {
		var session = sCache.getOrThrow(lab.getSession());
		model.addAttribute("lSession", session);

		if (session.getEdition() != null) {
			model.addAttribute("edition", eCache.getOrThrow(session.getEdition().getId()));
		} else {
			model.addAttribute("ec", ecCache.getOrThrow(session.getEditionCollection().getId()));
		}

		setLabEditingPageAttributes(session.getEditions().stream()
				.map(EditionSummaryDTO::getId).collect(Collectors.toList()), model);
	}

	/**
	 * Handles any {@link AccessDeniedException} thrown in this controller.
	 *
	 * @param  request The HTTP request associated with this exception.
	 *
	 * @return         The enrol page when the exception is thrown from a /lab/* page. Otherwise return the
	 *                 403 error page.
	 */
	@ExceptionHandler(AccessDeniedException.class)
	public String redirectToEnrolPage(HttpServletRequest request) {
		Pattern testPattern = Pattern.compile("/lab/([0-9]+)/*");
		Matcher matcher = testPattern.matcher(request.getRequestURI());
		if (matcher.matches() && matcher.groupCount() == 1) {
			Optional<Lab> lab = labRepository.findById(Long.parseLong(matcher.group(1)));
			var session = sCache.getOrThrow(lab.orElseThrow().getSession());
			var edition = session.getEdition().getId();
			return "redirect:/edition/" + edition + "/enrol";
		}
		return "error/403";
	}

}
