/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static nl.tudelft.labracore.api.dto.RolePersonDetailsDTO.TypeEnum.*;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.lib.security.user.AuthenticatedPerson;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.librador.resolver.annotations.PathEntity;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.dto.patch.FeedbackPatchDTO;
import nl.tudelft.queue.dto.patch.RequestPatchDTO;
import nl.tudelft.queue.dto.util.RequestTableFilterDTO;
import nl.tudelft.queue.dto.view.RequestViewDTO;
import nl.tudelft.queue.dto.view.RequestWithEventsViewDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.repository.RequestRepository;
import nl.tudelft.queue.service.*;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PageableDefault;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

@Controller
public class RequestController {

	@Autowired
	private RequestRepository rr;

	@Autowired
	private RequestService rs;

	@Autowired
	private LabService ls;

	@Autowired
	private RequestTableService rts;

	@Autowired
	private PersonControllerApi pApi;

	@Autowired
	private RoleDTOService rds;

	@Autowired
	private FeedbackService fs;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private PersonCacheManager pCache;

	@Autowired
	private SessionCacheManager sCache;

	/**
	 * Gets the view for the next request or a redirect to the requests page if no new request can be found.
	 *
	 * @param  lab    The lab that the person is requesting a new request for.
	 * @param  person The person that is requesting a new request to handle.
	 * @return        The redirect to either the request view or to the request table page.
	 */
	@GetMapping("/requests/next/{lab}")
	@PreAuthorize("@permissionService.canTakeRequest(#lab.id)")
	public String getNextRequest(@PathEntity Lab lab,
			@AuthenticatedPerson Person person) {
		RequestTableFilterDTO filter = rts.checkAndStoreFilterDTO(null, "/requests");
		Optional<Request> request = rs.takeNextRequest(person, lab, filter);

		return request.map(r -> "redirect:/request/" + r.getId()).orElse("redirect:/requests");
	}

	/**
	 * Gets the request table view page. This page should be accessible to assistants, TAs, teachers, etc.,
	 * but not for regular students. This page displays a big table containing all currently open requests.
	 * The table can be filtered to display less requests and a next request can be grabbed from this page.
	 *
	 * @param  pageable The pageable determining the current size and page of the view.
	 * @param  model    The model to fill out for Thymeleaf template resolution.
	 * @return          The Thymeleaf template to resolve.
	 */
	@GetMapping("/requests")
	@PreAuthorize("@permissionService.canViewRequests()")
	public String getRequestTableView(Model model,
			@AuthenticatedPerson Person assistant,
			@PageableDefault(sort = "createdAt", direction = Sort.Direction.DESC, size = 25) Pageable pageable) {
		var filter = rts.checkAndStoreFilterDTO(null, "/requests");

		List<Lab> labs = rts.addFilterAttributes(model, null);

		model.addAttribute("page", "requests");
		model.addAttribute("filter", filter);
		model.addAttribute("requests", rts.convertRequestsToView(
				rr.findAllByFilter(labs, filter, pageable)));
		model.addAttribute("requestCounts", rts.labRequestCounts(
				labs, assistant, filter));

		return "request/list";
	}

	/**
	 * Updates the information on the location of a student within the given request.
	 *
	 * @param  request The request to update location information on.
	 * @param  dto     The dto carrying the update information on the location.
	 * @return         A redirect back to the lab view page.
	 */
	@Transactional
	@PostMapping("/request/{request}/update-request-info")
	@PreAuthorize("@permissionService.canUpdateRequest(#request.id)")
	public String updateRequestInfo(@PathEntity Request request,
			RequestPatchDTO dto) {
		dto.apply(request);

		return "redirect:/lab/" + request.getLab().getId();
	}

	/**
	 * Gets the singular request view page. This page should be accessible to all those
	 *
	 * @param  request The request that is requested to be viewed.
	 * @param  model   The model to fill out for Thymeleaf template resolution.
	 * @return         The Thymeleaf template to resolve.
	 */
	@GetMapping("/request/{request}")
	@PreAuthorize("@permissionService.canViewRequest(#request.id)")
	public String getRequestView(@PathEntity Request request, Model model) {
		if (request.getEventInfo().getAssignedTo() != null) {
			model.addAttribute("assistant", pApi
					.getPersonById(request.getEventInfo().getAssignedTo()).block());
		}

		model.addAttribute("request", View.convert(request, RequestWithEventsViewDTO.class));
		model.addAttribute("prevRequests",
				View.convert(rr.findAllPreviousRequests(request), RequestViewDTO.class));
		ls.setOrganizationInModel(request.getLab(), model);

		return "request/view";
	}

	/**
	 * Gets the request approval view. This page is used to submit a request approval.
	 *
	 * @param  request The request that is requested to be approved.
	 * @param  model   The model to fill out for Thymeleaf template resolution.
	 * @return         The Thymeleaf template to resolve.
	 */
	@GetMapping("/request/{request}/approve")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String getRequestApproveView(@PathEntity Request request, Model model) {
		model.addAttribute("request", request);
		ls.setOrganizationInModel(request.getLab(), model);

		return "request/approve";
	}

	/**
	 * Performs the 'approve request' action for a request. This controller method calls the service method to
	 * handle a request approve command and redirects the user back to the requests page to grab their next
	 * request.
	 *
	 * @param  request            The request that the user approves.
	 * @param  person             The user that is approving the given request.
	 * @param  reasonForAssistant The reason the assistant gives other assistants for approving the request.
	 * @param  redirectAttributes Attributes used to fill in a model message upon redirection.
	 * @return                    A redirect to the requests table page.
	 */
	@PostMapping("/request/{request}/approve")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String approveRequest(@PathEntity Request request,
			@AuthenticatedPerson Person person,
			@RequestParam(value = "reasonForAssistant") String reasonForAssistant,
			RedirectAttributes redirectAttributes) {
		rs.approveRequest(request, person.getId(), reasonForAssistant);

		redirectAttributes.addFlashAttribute("message", "Request #" + request.getId() + " approved");

		return "redirect:/requests";
	}

	/**
	 * Gets the request reject view. This page is used to submit a request rejection.
	 *
	 * @param  request The request that is requested to be rejected.
	 * @param  model   The model to fill out for Thymeleaf template resolution.
	 * @return         The Thymeleaf template to resolve.
	 */
	@GetMapping("/request/{request}/reject")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String getRequestRejectView(@PathEntity Request request, Model model) {
		model.addAttribute("request", request);
		ls.setOrganizationInModel(request.getLab(), model);

		return "request/reject";
	}

	/**
	 * Performs the 'reject request' action for a request. This controller method calls the service method to
	 * handle a request reject command and redirects the user back to the requests page to grab their next
	 * request.
	 *
	 * @param  request            The request that the user rejects.
	 * @param  person             The user that is rejecting the given request.
	 * @param  reasonForAssistant The reason the assistant gives other assistants for rejecting the request.
	 * @param  reasonForStudent   The reason the assistant gives the student(s) for rejecting the request.
	 * @param  redirectAttributes Attributes used to fill in a model message upon redirection.
	 * @return                    A redirect to the requests table page.
	 */
	@PostMapping("/request/{request}/reject")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String rejectRequest(@PathEntity Request request,
			@AuthenticatedPerson Person person,
			@RequestParam(value = "reasonForAssistant") String reasonForAssistant,
			@RequestParam(value = "reasonForStudent") String reasonForStudent,
			RedirectAttributes redirectAttributes) {
		rs.rejectRequest(request, person.getId(), reasonForAssistant, reasonForStudent);

		redirectAttributes.addFlashAttribute("message", "Request #" + request.getId() + " rejected");

		return "redirect:/requests";
	}

	/**
	 * Gets the request forward view. This page is used to forward a request to a different TA.
	 *
	 * @param  request The request that is requested to be forwarded.
	 * @param  model   The model to fill out for Thymeleaf template resolution.
	 * @return         The Thymeleaf template to resolve.
	 */
	@GetMapping("/request/{request}/forward")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String getRequestForwardView(@AuthenticatedPerson Person user,
			@PathEntity Request request, Model model) {
		var session = sCache.getOrThrow(request.getLab().getSession());
		var assistants = session.getEditions().stream()
				.flatMap(e -> rds.roles(
						eCache.getOrThrow(e.getId()), Set.of(TA, HEAD_TA, TEACHER, TEACHER_RO)).stream())
				.filter(p -> !p.getId().equals(user.getId()))
				.collect(Collectors.toList());

		ls.setOrganizationInModel(request.getLab(), model);

		model.addAttribute("request", request);
		model.addAttribute("assistants", assistants);

		return "request/forward";
	}

	/**
	 * Performs the 'forward request' action for a request. This controller method calls the service method to
	 * handle a request forward command and redirects the user back to the requests page to grab their next
	 * request.
	 *
	 * @param  request            The request that the user forwards.
	 * @param  assistant          The assistant the request should be forwarded to.
	 * @param  reasonForAssistant The reason the assistant gives other assistants for forwarding the request.
	 * @param  redirectAttributes Attributes used to fill in a model message upon redirection.
	 * @return                    A redirect to the requests table page.
	 */
	@PostMapping("/request/{request}/forward")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String forwardRequest(@PathEntity Request request,
			@AuthenticatedPerson Person assistant,
			@RequestParam(value = "assistant") Long forwardedTo,
			@RequestParam(value = "reasonForAssistant") String reasonForAssistant,
			RedirectAttributes redirectAttributes) {
		if (forwardedTo == -1L) {
			rs.forwardRequestToAnyone(request, assistant, reasonForAssistant);
		} else {
			rs.forwardRequestToPerson(request, assistant, pCache.getOrThrow(forwardedTo), reasonForAssistant);
		}

		redirectAttributes.addFlashAttribute("message", "Request #" + request.getId() + " forwarded");

		return "redirect:/requests";
	}

	/**
	 * Makes a request not found status request for the specific request. This means the final request status
	 * will be changed to "Not Found" and the request will be labeled as such everywhere it can be seen.
	 *
	 * @param  request The request to mark "Not Found".
	 * @return         A redirect to the requests table page.
	 */
	@GetMapping("/request/{request}/not-found")
	@PreAuthorize("@permissionService.canFinishRequest(#request.id)")
	public String couldNotFindStudent(@PathEntity Request request,
			@AuthenticatedPerson Person person) {
		rs.couldNotFindStudent(request, person);

		return "redirect:/requests";
	}

	/**
	 * Allows teachers to choose a specific request to handle which is not yet being processed by a TA to
	 * allow out of order processing.
	 *
	 * @param  request The request the user wants to pick
	 * @param  person  The person who wants to pick the request
	 * @return         The request the user is currently processing. Either the picked request, or the request
	 *                 he/she was still processing.
	 */
	@GetMapping("/request/{request}/pick")
	@PreAuthorize("@permissionService.canPickRequest(#request.id)")
	public String pickRequest(@PathEntity Request request, @AuthenticatedPerson Person person) {
		Request currRequest = rs.pickRequest(person, request);
		return "redirect:/request/" + currRequest.getId();
	}

	/**
	 * Creates or changes the feedback of a group on a TA on the given request.
	 *
	 * @param  requestId   The id of the request on which the feedback is placed.
	 * @param  assistantId The id of the assistant on which the feedback is placed.
	 * @return             A redirect back to the same request overview.
	 */
	@PostMapping("/request/{request}/feedback/{assistantId}")
	@PreAuthorize("@permissionService.canGiveFeedback(#request.id)")
	public String giveFeedback(@PathEntity Request request,
			@PathVariable Long assistantId,
			FeedbackPatchDTO dto) {
		fs.updateFeedback(request, assistantId, dto);

		return "redirect:/request/" + request.getId();
	}

}
