/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.dto.create;

import java.util.*;
import java.util.stream.Collectors;

import lombok.*;
import lombok.experimental.SuperBuilder;
import nl.tudelft.labracore.api.dto.RoomSummaryDTO;
import nl.tudelft.labracore.api.dto.SessionDetailsDTO;
import nl.tudelft.librador.dto.create.Create;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.embeddables.Slot;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.RequestType;

@Data
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = false)
public abstract class LabCreateDTO<D extends Lab> extends Create<D> {
	private String name;
	private Slot slot;

	private CommunicationMethod communicationMethod;

	private Set<Long> modules = new HashSet<>();

	private Map<Long, Set<RequestType>> requestTypes = new HashMap<>();

	private Set<Long> rooms = new HashSet<>();

	public LabCreateDTO(SessionDetailsDTO session, Lab lab) {
		name = session.getName();
		slot = new Slot(session.getStart(), session.getEnd());

		communicationMethod = lab.getCommunicationMethod();

		modules = lab.getModules();

		requestTypes = lab.getAllowedRequests().stream()
				.collect(Collectors.groupingBy(AllowedRequest::getAssignment,
						Collectors.mapping(AllowedRequest::getType, Collectors.toSet())));

		rooms = session.getRooms().stream().map(RoomSummaryDTO::getId).collect(Collectors.toSet());
	}

	@Override
	public void validate() {
		nonEmpty("name", name);
		nonNull("slot", slot);

		nonNull("communicationMethod", communicationMethod);

		nonEmpty("modules", modules);

		nonEmpty("requestTypes", requestTypes);
		nonEmpty("rooms", rooms);
	}

	@Override
	protected void postApply(Lab data) {
		data.setAllowedRequestsFromMap(requestTypes);
	}

	protected void nonNull(String field, Object o) {
		if (o == null) {
			errors.rejectValue(field, "Field cannot be null");
		}
	}

	private void nonEmpty(String field, Collection<?> i) {
		if (i == null || i.isEmpty()) {
			errors.rejectValue(field, "Field cannot be null or empty");
		}
	}

	private void nonEmpty(String field, Map<?, ?> i) {
		if (i == null || i.isEmpty()) {
			errors.rejectValue(field, "Field cannot be null or empty");
		}
	}
}
