/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.dto.create;

import static nl.tudelft.librador.SpringContext.getBean;

import java.util.Objects;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import lombok.*;
import nl.tudelft.librador.dto.create.Create;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.dto.id.TimeSlotIdDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.SlottedLab;

@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = false)
public class RequestCreateDTO extends Create<Request> {
	@NotNull
	private RequestType requestType;

	@Size(max = 250)
	private String comment;

	@Size(max = 500)
	private String question;

	private TimeSlotIdDTO timeSlot;

	@NotNull
	private Long assignment;

	@NotNull
	private Long room;

	private transient Lab lab;

	@Override
	public void validate() {
		if (requestType == RequestType.QUESTION) {
			nonEmpty("question", question);
		}

		var session = getBean(SessionCacheManager.class).getOrThrow(lab.getSession());
		if (session.getRooms().stream().noneMatch(r -> Objects.equals(r.getId(), room))) {
			errors.rejectValue("room", "A room with id " + room + " is not available in the lab.");
		}

		if (!lab.getAllowedRequests().contains(new AllowedRequest(assignment, requestType))) {
			errors.rejectValue("assignment", "Incompatible assignment for request type " + requestType);
		}

		if (lab instanceof SlottedLab &&
				(timeSlot == null || !Objects.equals(timeSlot.apply().getLab().getId(), lab.getId()) ||
						timeSlot.apply().isFull())) {
			errors.rejectValue("slot", "No time slot selected for request");
		}
	}

	@Override
	public Class<Request> clazz() {
		return Request.class;
	}
}
