/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.dto.view;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import nl.tudelft.labracore.api.dto.ModuleDetailsDTO;
import nl.tudelft.labracore.api.dto.SessionDetailsDTO;
import nl.tudelft.librador.SpringContext;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.cache.ModuleCacheManager;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.LabType;

@Data
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = false)
public class LabViewDTO<D extends Lab> extends View<D> {
	private LabType type;

	private Long id;

	private SessionDetailsDTO session;

	private CommunicationMethod communicationMethod;

	private Boolean enqueueClosed;
	private LocalDateTime deletedAt;

	private Set<AllowedRequest> allowedRequests;
	private Set<ModuleDetailsDTO> modules;

	private List<Request> requests = new ArrayList<>();

	@Override
	public void postApply() {
		super.postApply();

		session = SpringContext.getBean(SessionCacheManager.class).getOrThrow(data.getSession());
		modules = new HashSet<>(
				SpringContext.getBean(ModuleCacheManager.class).get(data.getModules().stream()));
	}

	/**
	 * @return Whether this lab is currently open.
	 */
	public boolean isOpen() {
		var now = LocalDateTime.now();
		return session.getStart().isBefore(now) && session.getEnd().isAfter(now);
	}

	/**
	 * @return The underlying lab data.
	 */
	public D getData() {
		return data;
	}
}
