/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.dto.view;

import static nl.tudelft.librador.SpringContext.getBean;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.cache.*;
import nl.tudelft.queue.model.Feedback;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.TimeSlot;
import nl.tudelft.queue.model.enums.RequestType;

import com.beust.jcommander.Strings;

@Data
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = false)
public class RequestViewDTO extends View<Request> {
	private Long id;

	private LocalDateTime createdAt;

	private String comment;
	private String question;

	private RequestEventInfoViewDTO eventInfo;

	private RequestType requestType;
	private AssignmentDetailsDTO assignment;

	private RoomSummaryDTO room;
	private String jitsiRoom;
	private TimeSlot slot;

	private PersonSummaryDTO requester;
	private StudentGroupDetailsDTO studentGroup;

	private Lab lab;
	private SessionDetailsDTO session;

	private List<Feedback> feedbacks;

	@Override
	public void postApply() {
		assignment = getBean(AssignmentCacheManager.class).getOrThrow(data.getAssignment());
		room = getBean(RoomCacheManager.class).getOrThrow(data.getRoom());
		requester = getBean(PersonCacheManager.class).getOrThrow(data.getRequester());

		session = getBean(SessionCacheManager.class).getOrThrow(lab.getSession());
		studentGroup = getBean(StudentGroupCacheManager.class).getOrThrow(data.getStudentGroup());

		eventInfo.postApply();
	}

	/**
	 * Gets the name of the student group that this request is from.
	 *
	 * @return The name of the group that the request was created for with description of its members.
	 */
	public String studentGroupName() {
		return "Group " + studentGroup.getName() + " (" +
				Strings.join(", ", studentGroup.getMembers().stream()
						.filter(r -> r.getType() == RolePersonLayer1DTO.TypeEnum.STUDENT)
						.map(r -> r.getPerson().getDisplayName()).collect(Collectors.toList()))
				+
				")";
	}

	/**
	 * Converts the student group name or requester name into a string.
	 *
	 * @return The name of the requesting entity (either the group or the student).
	 */
	public String requesterEntityName() {
		if (studentGroup.getMemberUsernames().size() == 1) {
			return requester.getDisplayName();
		} else {
			return studentGroupName();
		}
	}

	/**
	 * @return The name of the organization that manages the session this Request is a part of. (This could
	 *         either be an EditionCollection name or an Edition name)
	 */
	public String organizationName() {
		if (session.getEditionCollection() != null) {
			return session.getEditionCollection().getName();
		}
		return session.getEdition().getName();
	}

	/**
	 * Converts the request into a sentence readable for humans.
	 *
	 * @return The human-readable sentence representing this request.
	 */
	public String toSentence() {
		StringBuilder builder = new StringBuilder(requesterEntityName())
				.append(" ")
				.append(requestType.toSentence())
				.append(" ")
				.append(assignment.getName());

		if (slot != null) {
			builder.append(" at ")
					.append(slot.getSlot().toSentence());
		}

		return builder.toString();
	}
}
