/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model;

import java.io.Serializable;
import java.time.LocalDateTime;

import javax.persistence.*;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * Entity class representing feedback given to a TA about one request they contributed to handling. Feedback
 * may be given multiple times per request, but can only given once per TA per request. In other words, one
 * can give feedback on every TA that participated on a request.
 */
@Data
@Entity
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class Feedback {
	/**
	 * The embeddable id of a Feedback object.
	 */
	@Data
	@Embeddable
	@NoArgsConstructor
	@AllArgsConstructor
	public static class Id implements Serializable {
		private static final long serialVersionUID = -1299374329281277135L;

		/**
		 * The id of the request that this feedback is placed on.
		 */
		@NotNull
		private Long requestId;

		/**
		 * The id of the assistant this feedback is about.
		 */
		@NotNull
		private Long assistantId;
	}

	/**
	 * The embedded ID of this Feedback entity.
	 */
	@NotNull
	@EmbeddedId
	private Id id;

	/**
	 * The request from which feedback is given. Feedback is assumed to be given by the request entity
	 * (student group) placing the original request.
	 */
	@ManyToOne
	@MapsId("requestId")
	private Request request;

	/**
	 * The feedback provided by the request group in words.
	 */
	@Size(max = 250)
	private String feedback;

	/**
	 * The rating given by the request group to the assistant.
	 */
	@Min(1)
	@Max(5)
	@NotNull
	private Integer rating;

	/**
	 * The time that this feedback was placed on the site. (not the last updated time, but the time that the
	 * feedback first appeared).
	 */
	@NotNull
	private LocalDateTime createdAt;

	/**
	 * The time that this feedback was last updated by the student group.
	 */
	@NotNull
	private LocalDateTime lastUpdatedAt;
}
