/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model;

import static java.time.LocalDateTime.now;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.persistence.*;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

import lombok.*;
import lombok.experimental.SuperBuilder;
import nl.tudelft.labracore.api.dto.SessionDetailsDTO;
import nl.tudelft.queue.dto.create.LabCreateDTO;
import nl.tudelft.queue.dto.patch.LabPatchDTO;
import nl.tudelft.queue.dto.view.LabViewDTO;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.LabType;
import nl.tudelft.queue.model.enums.RequestType;

import org.hibernate.annotations.SQLDelete;
import org.hibernate.annotations.Where;
import org.hibernate.validator.constraints.UniqueElements;

@Data
@Entity
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
@DiscriminatorColumn(name = "type")
@Inheritance(strategy = InheritanceType.JOINED)
@SQLDelete(sql = "UPDATE lab SET deleted_at = NOW(), slot_id = NULL WHERE id = ?")
@Where(clause = "deleted_at IS NULL")
public abstract class Lab {
	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	/**
	 * The type of the lab.
	 */
	@NotNull
	@Column(name = "type", insertable = false, updatable = false)
	@Enumerated(EnumType.STRING)
	private LabType type = getLabType();

	/**
	 * The session that this lab is related to. The session can be seen as a sort of superclass to this class
	 * as sessions contain all the necessary information about rooms, assignments, etc. needed for a lab to be
	 * displayed in different applications.
	 */
	@NotNull
	private Long session;

	/**
	 * The communication method that will be used for the lab. This could range from physical contact to Jitsi
	 * meetings.
	 */
	@NotNull
	@Enumerated(EnumType.STRING)
	private CommunicationMethod communicationMethod;

	/**
	 * Whether enqueueing to the lab is (manually) closed. This can be toggled by teachers to close exam lab
	 * slot selection or as an all-or-nothing means to end a busy lab early.
	 */
	@NotNull
	@Builder.Default
	private Boolean enqueueClosed = false;

	/**
	 * The time this lab was deleted, or null if this lab was never deleted. This field is used as a means to
	 * soft-delete labs.
	 */
	@Builder.Default
	private LocalDateTime deletedAt = null;

	/**
	 * The request types that are allowed within this lab.
	 */
	@NotEmpty
	@UniqueElements
	@Builder.Default
	@ElementCollection
	private Set<AllowedRequest> allowedRequests = new HashSet<>();

	/**
	 * The list of modules that this lab accommodates to.
	 */
	@NotEmpty
	@UniqueElements
	@Builder.Default
	@ElementCollection
	private Set<Long> modules = new HashSet<>();

	/**
	 * The mapped list of requests that are created for this lab.
	 */
	@Builder.Default
	@ToString.Exclude
	@EqualsAndHashCode.Exclude
	@OneToMany(mappedBy = "lab", cascade = { CascadeType.ALL })
	private List<Request> requests = new ArrayList<>();

	/**
	 * Sets the allowed request types for this lab using a map of assignments to the request types that are
	 * allowed to be created for that assignment.
	 *
	 * @param requestTypes The mapping of assignments to request types from which to read allowed requests.
	 */
	public void setAllowedRequestsFromMap(Map<Long, Set<RequestType>> requestTypes) {
		this.setAllowedRequests(requestTypes.entrySet().stream()
				.flatMap(e -> {
					if (e.getValue() == null) {
						return Stream.empty();
					} else {
						return e.getValue().stream()
								.filter(Objects::nonNull)
								.map(v -> new AllowedRequest(e.getKey(), v));
					}
				})
				.collect(Collectors.toSet()));
	}

	/**
	 * Gets the currently open request for the given person.
	 *
	 * @param  personId The id of the person to lookup an open request for.
	 * @return          An optional either containing the current request for the person or nothing.
	 */
	public Optional<Request> getOpenRequestForPerson(Long personId) {
		return requests.stream()
				.filter(r -> !r.getEventInfo().getStatus().isFinished() && personId.equals(r.getRequester()))
				.findFirst();
	}

	/**
	 * Gets the currently pending request for the given person.
	 *
	 * @param  personId The id of the person to lookup a pending request for.
	 * @return          An optional either containing the current request for the person or nothing.
	 */
	public Optional<Request> getPendingRequestForPerson(Long personId) {
		return getQueue().stream()
				.filter(r -> r.getEventInfo().getStatus().isPending() && personId.equals(r.getRequester()))
				.findFirst();
	}

	/**
	 * Gets all requests for the given person within this lab.
	 *
	 * @param  personId The id of the person to lookup requests for.
	 * @return          A list of all requests for the person within this lab.
	 */
	public List<Request> getAllRequestsForPerson(Long personId) {
		return requests.stream()
				.filter(r -> personId.equals(r.getRequester()))
				.collect(Collectors.toList());
	}

	/**
	 * Checks whether this lab contains an open request from the person with the given id.
	 *
	 * @param  personId The id of the person to check for.
	 * @return          Whether a request is currently open for the given person.
	 */
	public boolean hasOpenRequestForPerson(Long personId) {
		return getOpenRequestForPerson(personId).isPresent();
	}

	/**
	 * Gets the position a person currently has in the queue, or -1 if the person cannot be found in the
	 * queue.
	 *
	 * @param  personId The id of the person to check the position of.
	 * @return          The position of the person in the current lab queue.
	 */
	public int position(Long personId) {
		List<Request> queue = getQueue();
		return queue.stream().filter(r -> personId.equals(r.getRequester())).findFirst()
				.map(queue::indexOf)
				.orElse(-1) + 1;
	}

	/**
	 * Gets the average waiting time experienced by students in this lab over the last hour.
	 *
	 * @return The average waiting time if one can be calculated, for fresh labs this is empty.
	 */
	public OptionalDouble averageWaitingTime() {
		return getHandled().stream()
				.filter(r -> r.getEventInfo().getLastEventAt().isAfter(now().minusHours(1)))
				.flatMapToDouble(r -> r.waitingTime().stream())
				.average();
	}

	/**
	 * Filters out finished requests and assembles a sorted list representing the current queue.
	 *
	 * @return The list of requests representing the current queue.
	 */
	public List<Request> getQueue() {
		return requests.stream()
				.filter(request -> request.getEventInfo().getStatus().isPending())
				.sorted(Comparator.comparing(Request::getCreatedAt))
				.collect(Collectors.toUnmodifiableList());
	}

	/**
	 * Filters out unhandled open requests and assembles a list of all handled requests.
	 *
	 * @return The list of all requests handled by a TA.
	 */
	public List<Request> getHandled() {
		return requests.stream()
				.filter(request -> request.getEventInfo().getStatus().isHandled())
				.collect(Collectors.toList());
	}

	/**
	 * Gets the type of this lab.
	 *
	 * @return The type of this lab.
	 */
	protected abstract LabType getLabType();

	/**
	 * Copies this Lab object into a new {@link LabCreateDTO} of the right type.
	 *
	 * @param  session The session that this Lab is connected to.
	 * @return         A copy of this Lab object in the form of a create DTO.
	 */
	public abstract LabCreateDTO<?> copyLabCreateDTO(SessionDetailsDTO session);

	/**
	 * @return A newly constructed instance of a {@link LabPatchDTO} subclass depending on the type of the
	 *         lab.
	 */
	public abstract LabPatchDTO<?> newPatchDTO();

	/**
	 * @return A newly constructed {@link LabViewDTO} constructed based on this Lab.
	 */
	public abstract LabViewDTO<?> toViewDTO();
}
