/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.OptionalDouble;

import javax.persistence.*;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;

import lombok.*;
import nl.tudelft.queue.model.embeddables.RequestEventInfo;
import nl.tudelft.queue.model.enums.RequestType;

/**
 * A request that has been made by a StudentGroup, that the database representation of someone asking for
 * help. Part of this class stores it's data directly in the Request table. The lifecycle of a Request, that
 * is being: created, reassigned, forwarded, approved, etc; is done through event sourcing
 * {@link RequestEvent}. Every change in the requests lifecycle append a new event to this log.
 * <p>
 * The entire log is then reran when the Request is recreated by hibernate through an {@link PostLoad} hook,
 * which fills in the remaining transient (not persisted) fields of this class.
 */
@Data
@Entity
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class Request {
	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	private Long id;

	/**
	 * The type of the request (either a question or a submission).
	 */
	@NotNull
	private RequestType requestType;

	/**
	 * The comment a student placed describing their current situation or how to find them.
	 */
	@Size(max = 250)
	private String comment;

	/**
	 * The question a student placed to describe the intent of their request.
	 */
	@Lob
	@Column
	@Size(max = 500)
	private String question;

	/**
	 * The moment this request was created.
	 */
	@NotNull
	@Builder.Default
	private LocalDateTime createdAt = LocalDateTime.now();

	/**
	 * The Jitsi room this request will take place in.
	 */
	private String jitsiRoom;

	/**
	 * The timeslot this request occupies. Timeslots can be used as opposed to direct requests to allow for
	 * reserving a TA for a specific time ahead of the lab starting.
	 */
	@ManyToOne
	private TimeSlot timeSlot;

	/**
	 * The lab this request takes part in.
	 */
	@ManyToOne
	private Lab lab;

	/**
	 * The Assignment that this request is asking a question about or checking a submission of.
	 */
	@NotNull
	private Long assignment;

	/**
	 * The room the student group is in and wants to get a TA sent to.
	 */
	@NotNull
	private Long room;

	/**
	 * The information about the events that occurred for this request, summarized in an embeddable aggregate
	 * object.
	 */
	@Embedded
	@Builder.Default
	private RequestEventInfo eventInfo = new RequestEventInfo();

	/**
	 * The person that created the request in the name of their student group.
	 */
	@NotNull
	private Long requester;

	/**
	 * The student group for which the request was made.
	 */
	@NotNull
	private Long studentGroup;

	/**
	 * The feedback given through this Request.
	 */
	@Builder.Default
	@ToString.Exclude
	@EqualsAndHashCode.Exclude
	@OneToMany(mappedBy = "request")
	private List<Feedback> feedbacks = new ArrayList<>();

	/**
	 * Gets the waiting time experienced during this request. The waiting time is defined as the time between
	 * the first time that the request was processed and the creation time of the request.
	 *
	 * @return The waiting time experienced during this request.
	 */
	public OptionalDouble waitingTime() {
		if (eventInfo.getFirstProcessedAt() == null || createdAt == null) {
			return OptionalDouble.empty();
		}

		return OptionalDouble
				.of(ChronoUnit.SECONDS.between(createdAt, eventInfo.getFirstProcessedAt()));
	}
}
