/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model.embeddables;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

import javax.persistence.*;
import javax.validation.constraints.NotNull;

import lombok.*;
import nl.tudelft.queue.cqsr.Aggregate;
import nl.tudelft.queue.model.RequestEvent;
import nl.tudelft.queue.model.enums.RequestStatus;

@Data
@Builder
@Embeddable
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class RequestEventInfo extends Aggregate<RequestEventInfo, RequestEvent> {
	/**
	 * The status of the request, filled in by event application.
	 */
	@NotNull
	@Builder.Default
	@Enumerated(EnumType.STRING)
	private RequestStatus status = RequestStatus.PENDING;

	/**
	 * The reason for the current status, filled in by event application.
	 */
	private String reasonForAssistant;

	/**
	 * The reason for the current status visible for a student, filled in by event application.
	 */
	private String reasonForStudent;

	/**
	 * The last time an event occurred to this request, filled in by event application.
	 */
	private LocalDateTime lastEventAt;

	/**
	 * The last time that the associated request was assigned to a TA.
	 */
	private LocalDateTime lastAssignedAt;

	/**
	 * The first time that this request was to be processed.
	 */
	private LocalDateTime firstProcessedAt;

	/**
	 * The time that the associated request was handled (approved, rejected or not found).
	 */
	private LocalDateTime handledAt;

	/**
	 * The time that the associated request was finished (approved, rejected, revoked or not found).
	 */
	private LocalDateTime finishedAt;

	/**
	 * The Assistant that this request is currently forwarded to, filled in by event application.
	 */
	private Long forwardedTo;

	/**
	 * The Assistant that this request is currently forwarded by, filled in by event application.
	 */
	private Long forwardedBy;

	/**
	 * The Assistant currently helping with this request, filled in by event application.
	 */
	private Long assignedTo;

	/**
	 * The events that occurred on this Request.
	 */
	@Builder.Default
	@ToString.Exclude
	@OrderBy("timestamp asc")
	@EqualsAndHashCode.Exclude
	@OneToMany(cascade = { CascadeType.ALL }, mappedBy = "request")
	private List<RequestEvent> events = new ArrayList<>();

	@Override
	public List<? extends RequestEvent> events() {
		return events;
	}
}
