/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.repository;

import static com.querydsl.jpa.JPAExpressions.select;
import static java.time.LocalDateTime.now;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.queue.dto.util.RequestTableFilterDTO;
import nl.tudelft.queue.model.*;
import nl.tudelft.queue.model.enums.LabType;
import nl.tudelft.queue.model.enums.RequestStatus;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.querydsl.QuerydslPredicateExecutor;
import org.springframework.data.repository.CrudRepository;
import org.springframework.lang.NonNull;

import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.BooleanExpression;

public interface RequestRepository extends CrudRepository<Request, Long>, QuerydslPredicateExecutor<Request> {
	QRequest qr = QRequest.request;

	@NonNull
	@Override
	List<Request> findAll(@NonNull Predicate predicate);

	@NonNull
	@Override
	List<Request> findAll(@NonNull Predicate predicate, @NonNull Sort sort);

	/**
	 * Finds all requests that are part of the given lab.
	 *
	 * @param  lab The lab to get the requests from.
	 * @return     The list of requests in the given lab.
	 */
	default List<Request> findAllByLab(Lab lab) {
		return findAll(qr.lab.id.eq(lab.getId()));
	}

	/**
	 * Finds a page of previous requests from the given list of groups after applying the given filter.
	 *
	 * @param  groups   The groups to search for in made requests.
	 * @param  filter   The filter to apply to the list of found requests.
	 * @param  pageable The pageable containing configurations for displaying the page.
	 * @return          The page of previous requests.
	 */
	default Page<Request> findAllPreviousRequests(List<Long> groups, RequestTableFilterDTO filter,
			Pageable pageable) {
		return findAll(createFilterBooleanExpression(filter, qr.studentGroup.in(groups)), pageable);
	}

	/**
	 * Finds all previous requests that have to do with the same assignment as the given request.
	 *
	 * @param  request The request to find historically similar requests for.
	 * @return         The sorted list of all previous related requests.
	 */
	default List<Request> findAllPreviousRequests(Request request) {
		return findAll(qr.assignment.eq(request.getAssignment())
				.and(qr.createdAt.before(request.getCreatedAt()))
				.and(qr.studentGroup.eq(request.getStudentGroup())),
				Sort.sort(Request.class).by(Request::getCreatedAt).ascending());
	}

	/**
	 * Finds the request that is currently being processed by the given assistant in the given lab if any
	 * exists.
	 *
	 * @param  assistant The assistant for which to look up the request.
	 * @param  lab       The lab in which to find the request.
	 * @return           Either the currently processing request or nothing.
	 */
	default Optional<Request> findCurrentlyProcessingRequest(Person assistant, Lab lab) {
		return findOne(qr.lab.id.eq(lab.getId())
				.and(qr.eventInfo.status.eq(RequestStatus.PROCESSING))
				.and(qr.eventInfo.assignedTo.eq(assistant.getId())));
	}

	/**
	 * Finds the request that is currently being processed by the given assistant in the given slot if any
	 * exists.
	 *
	 * @param  assistant The assistant for which to look up the request.
	 * @param  slot      The slot in which to find the request.
	 * @return           Either the currently processing request or nothing.
	 */
	default Optional<Request> findCurrentlyProcessingRequest(Person assistant, TimeSlot slot) {
		return findOne(qr.timeSlot.id.eq(slot.getId())
				.and(qr.eventInfo.status.eq(RequestStatus.PROCESSING))
				.and(qr.eventInfo.assignedTo.eq(assistant.getId())));
	}

	/**
	 * Finds the first request that is currently specifically forwarded to the given assistant within the
	 * given lab. This is useful for determining what request should be handled next.
	 *
	 * @param  assistant The assistant that the request should be forwarded to.
	 * @param  lab       The lab in which the request exists.
	 * @return           The forwarded request or none if no such request exists.
	 */
	default Optional<Request> findCurrentlyForwardedRequest(Person assistant, Lab lab) {
		return findAll(qr.lab.id.eq(lab.getId())
				.and(qr.eventInfo.status.eq(RequestStatus.FORWARDED))
				.and(qr.eventInfo.forwardedTo.eq(assistant.getId())),
				Sort.sort(Request.class).by((Request r) -> r.getEventInfo().getLastEventAt()).ascending())
						.stream().findFirst();
	}

	/**
	 * Finds the first request that is currently specifically forwarded to the given assistant within the
	 * given time slot.
	 *
	 * @param  assistant The assistant that is getting a next request.
	 * @param  slot      The slot for which the assistant is getting the next request.
	 * @return           The found request or none if no such request exists.
	 */
	default Optional<Request> findCurrentlyForwardedRequest(Person assistant, TimeSlot slot) {
		return findAll(qr.timeSlot.id.eq(slot.getId())
				.and(qr.eventInfo.status.eq(RequestStatus.FORWARDED))
				.and(qr.eventInfo.forwardedTo.eq(assistant.getId())),
				Sort.sort(Request.class).by((Request r) -> r.getEventInfo().getLastEventAt()).ascending())
						.stream().findFirst();
	}

	/**
	 * Finds all next requests for the given exam time slot.
	 *
	 * @param  timeSlot  The time slot to get next requests from.
	 * @param  assistant The assistant that is getting the next request.
	 * @return           All requests that can be picked next in the given time slot.
	 */
	default List<Request> findNextExamRequests(ClosableTimeSlot timeSlot, Person assistant) {
		return findAll(qr.timeSlot.id.eq(timeSlot.getId())
				.and(isStatusOrForwardedToAny(RequestStatus.PENDING, assistant)));
	}

	/**
	 * Counts the number of requests the assistant can grab next for the given time slot.
	 *
	 * @param  timeSlot  The id of the time slot the assistant would get the next request from.
	 * @param  assistant The assistant that is fetching the next request.
	 * @return           The number of requests that can be picked from the time slot.
	 */
	default long countNextExamRequests(Long timeSlot, Person assistant) {
		return count(qr.timeSlot.id.eq(timeSlot)
				.and(isStatusOrForwarded(RequestStatus.PENDING, assistant)));
	}

	/**
	 * Finds the next request within a timeslotted lab that has the PENDING or FORWARDED (to any) status. This
	 * request will be assigned to the assistant querying for a new request to handle. A timeslotted request
	 * can be taken if its slot is open or if its slot is 10 minutes from now.
	 *
	 * @param  lab       The timeslotted lab from which the next request will be taken.
	 * @param  assistant The assistant that is getting the next request.
	 * @param  filter    The filter to apply to the request table query.
	 * @return           The next slotted request from the given lab.
	 */
	default Optional<Request> findNextSlotRequest(Lab lab, Person assistant, RequestTableFilterDTO filter) {
		return findAll(createFilterBooleanExpression(filter,
				qr.lab.id.eq(lab.getId())
						.and(isStatusOrForwardedToAny(RequestStatus.PENDING, assistant))
						.and(qr.timeSlot.slot.opensAt.before(now().plusMinutes(10)))),
				PageRequest.of(0, 1, Sort.sort(Request.class)
						.by((Request r) -> r.getTimeSlot().getSlot().getOpensAt())
						.ascending()))
								.stream().findFirst();
	}

	/**
	 * Counts the number of slotted requests currently open in the given lab from the viewpoint of the given
	 * Person applying the given filter.
	 *
	 * @param  lab       The lab to find the number of open slotted requests for.
	 * @param  assistant The assistant requesting the number of open requests.
	 * @param  filter    The fitler the assistant is applying to their requests view.
	 * @return           The count of open slotted requests.
	 */
	default long countOpenSlotRequests(Lab lab, Person assistant, RequestTableFilterDTO filter) {
		return count(createFilterBooleanExpression(filter,
				qr.lab.id.eq(lab.getId())
						.and(isStatusOrForwarded(RequestStatus.PENDING, assistant))
						.and(qr.timeSlot.slot.opensAt.before(now().plusMinutes(10)))));
	}

	/**
	 * Finds the next request within a regular lab that has the PENDING or FORWARDED (to any) status. This
	 * request will be assigned to the assistant querying for a new request to handle.
	 *
	 * @param  lab       The lab from which the next request will be taken.
	 * @param  assistant The assistant that is getting the next request.
	 * @param  filter    The filter to apply to the request table query.
	 * @return           The next request from the given lab.
	 */
	default Optional<Request> findNextNormalRequest(Lab lab, Person assistant, RequestTableFilterDTO filter) {
		return findAll(createFilterBooleanExpression(filter,
				qr.lab.id.eq(lab.getId())
						.and(isStatusOrForwardedToAny(RequestStatus.PENDING, assistant))),
				PageRequest.of(0, 1, Sort.sort(Request.class).by(Request::getCreatedAt).ascending()))
						.stream().findFirst();
	}

	/**
	 * Counts the number of open regular requests in the given lab from the viewpoint of the given assistant
	 * with the given filter applied.
	 *
	 * @param  lab       The lab to count the number of open requests in.
	 * @param  assistant The assistant requesting this count.
	 * @param  filter    The filter applied by the assistant on the page they are requesting.
	 * @return           The number of regular requests in the given lab.
	 */
	default long countNormalRequests(Lab lab, Person assistant, RequestTableFilterDTO filter) {
		return count(createFilterBooleanExpression(filter,
				qr.lab.id.eq(lab.getId())
						.and(isStatusOrForwarded(RequestStatus.PENDING, assistant))));
	}

	/**
	 * Creates a {@link BooleanExpression} for querying the request table. This expression checks whether the
	 * request either has the given status or the forwarded status with no specific person it is forwarded to.
	 *
	 * @param  status The status that the request is desired to have.
	 * @return        The predicate that represents this expression.
	 */
	default BooleanExpression isStatusOrForwardedToAny(RequestStatus status, Person assistant) {
		return qr.eventInfo.status.eq(status)
				.or(qr.eventInfo.status.eq(RequestStatus.FORWARDED)
						.and(qr.eventInfo.forwardedTo.isNull())
						.and(qr.eventInfo.forwardedBy.ne(assistant.getId())));
	}

	/**
	 * Creates a {@link BooleanExpression} for querying the request table. This expressiopn check whether the
	 * request either has the given status or the forwarded status with a specific person or the forwarded
	 * status with no specific person it is forwarded to.
	 *
	 * @param  status    The status that the request is desired to have.
	 * @param  assistant The assistant to find forwarded requests for.
	 * @return           The predicate expression representing this condition.
	 */
	default BooleanExpression isStatusOrForwarded(RequestStatus status, Person assistant) {
		return qr.eventInfo.status.eq(status)
				.or(qr.eventInfo.status.eq(RequestStatus.FORWARDED)
						.andAnyOf(qr.eventInfo.forwardedTo.isNull()
								.and(qr.eventInfo.forwardedBy.ne(assistant.getId())),
								qr.eventInfo.forwardedTo.eq(assistant.getId())));
	}

	/**
	 * Finds all requests that pass the given filter. This filter is a DTO transferred from the page
	 * requesting to see these requests. The filter contains all labs, rooms, assignments, etc. that need to
	 * be displayed. If a field in the filter is left as an empty set, the filter ignores it.
	 *
	 * @param  labs     The labs that the should also be kept in the filter.
	 * @param  filter   The filter to apply to the boolean expression.
	 * @param  pageable The pageable object representing the page.
	 * @return          The filtered list of requests.
	 */
	default Page<Request> findAllByFilter(List<Lab> labs, RequestTableFilterDTO filter, Pageable pageable) {
		return findAll(qr.in(select(qr).from(qr)
				.leftJoin(qr.timeSlot, QTimeSlot.timeSlot).on(qr.timeSlot.id.eq(QTimeSlot.timeSlot.id))
				.where(createFilterBooleanExpression(filter, qr.lab.in(labs).and(
						qr.lab.type.eq(LabType.REGULAR)
								.or(qr.lab.type.in(LabType.SLOTTED, LabType.EXAM)
										.and(qr.timeSlot.slot.opensAt.before(now().plusMinutes(10)))))))),
				pageable);
	}

	/**
	 * Creates the boolean expression that will filter requests through a filter DTO. This filter is a DTO
	 * transferred from the page requesting to see these requests. The filter contains all labs, rooms,
	 * assignments, etc. that need to be displayed. If a field in the filter is left as an empty set, the
	 * filter ignores it.
	 *
	 * @param  filter The filter to apply to the boolean expression.
	 * @param  e      The boolean expression upon which the filter is applied.
	 * @return        The combined boolean expression of the original expression and the filter applied to it.
	 */
	default BooleanExpression createFilterBooleanExpression(RequestTableFilterDTO filter,
			BooleanExpression e) {
		e = and(e, filter.getAssigned(), qr.eventInfo.assignedTo::in);
		e = and(e, filter.getLabs(), qr.lab.id::in);
		e = and(e, filter.getRooms(), qr.room::in);
		e = and(e, filter.getAssignments(), qr.assignment::in);
		e = and(e, filter.getRequestStatuses(), qr.eventInfo.status::in);
		e = and(e, filter.getRequestTypes(), qr.requestType::in);

		return e;
	}

	/**
	 * Performs a conjunction between the two boolean expressions only if the given filter set is non-empty.
	 * If the filter set is empty, the original boolean expression is returned.
	 *
	 * @param  e   The boolean expression to build on.
	 * @param  t   The set of filter types.
	 * @param  f   The function that will create a new boolean expression from the filter types.
	 * @param  <T> The type of the filter types (usually a Long).
	 * @return     The combined boolean expression.
	 */
	default <T> BooleanExpression and(BooleanExpression e, Set<T> t,
			Function<Set<T>, BooleanExpression> f) {
		if (t.isEmpty()) {
			return e;
		}
		return e.and(f.apply(t));
	}
}
