/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import static java.time.LocalDateTime.now;

import java.util.Comparator;
import java.util.List;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.dto.PersonSummaryDTO;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.dto.patch.FeedbackPatchDTO;
import nl.tudelft.queue.model.Feedback;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.RequestEvent;
import nl.tudelft.queue.model.events.EventWithAssistant;
import nl.tudelft.queue.repository.FeedbackRepository;
import nl.tudelft.queue.repository.RequestRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class FeedbackService {

	@Autowired
	private PersonCacheManager pCache;

	@Autowired
	private FeedbackRepository fr;

	@Autowired
	private RequestRepository rr;

	/**
	 * Finds all assistant that are involved in the given request. This list of people is generated for
	 * students to be able to give feedback on the one TA they want to give feedback on. This list is ordered
	 * by event occurrence: the last occurred event will see its assistant at the top of the list.
	 *
	 * @param  requestId The id of the request to find all assistants for.
	 * @return           A list of person summaries representing the various assistants helping out with the
	 *                   given request.
	 */
	public List<PersonSummaryDTO> assistantInvolvedInRequest(Long requestId) {
		return pCache.get(rr.findById(requestId).stream()
				.flatMap(request -> request.getEventInfo().getEvents().stream())
				.sorted(Comparator.comparing(RequestEvent::getTimestamp).reversed())
				.filter(event -> event instanceof EventWithAssistant)
				.map(event -> ((EventWithAssistant) event).getAssistant())
				.distinct());
	}

	/**
	 * Updates feedback by the given request id and assistant id, but only if a feedback is already in the
	 * database by these ids. If none is found, a new feedback object is saved to the database and the given
	 * rating and feedback string are filled in in it.
	 *
	 * @param requestId   The id of the request the feedback is on.
	 * @param assistantId The id of the assistant the feedback is on.
	 * @param dto         The transferred Patch DTO representing the feedback.
	 */
	@Transactional
	public void updateFeedback(Request request, Long assistantId, FeedbackPatchDTO dto) {
		fr.findById(request.getId(), assistantId)
				.map(dto::apply)
				.orElseGet(() -> fr.save(Feedback.builder()
						.id(new Feedback.Id(request.getId(), assistantId))
						.request(request)
						.feedback(dto.getFeedback())
						.rating(dto.getRating())
						.createdAt(now())
						.lastUpdatedAt(now())
						.build()));
	}

}
