/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.SessionControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.EditionCollectionCacheManager;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.dto.create.LabCreateDTO;
import nl.tudelft.queue.dto.create.labs.AbstractSlottedLabCreateDTO;
import nl.tudelft.queue.dto.patch.LabPatchDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.AbstractSlottedLab;
import nl.tudelft.queue.repository.LabRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.ui.Model;

@Service
public class LabService {
	@Autowired
	private LabRepository lr;

	@Autowired
	private SessionControllerApi sApi;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private EditionCollectionCacheManager ecCache;

	@Autowired
	private SessionCacheManager sCache;

	@Autowired
	private TimeSlotService tss;

	public enum LabType {
		REGULAR,
		SHARED
	}

	/**
	 * Sets the organizational unit that the given lab relates to in the given model. This could either be an
	 * Edition or an EditionCollection.
	 *
	 * @param lab   The lab from which to extract the organizational unit.
	 * @param model The model to set for Thymeleaf resolution.
	 */
	public void setOrganizationInModel(Lab lab, Model model) {
		var session = sCache.getOrThrow(lab.getSession());
		setOrganizationInModel(session, model);
	}

	/**
	 * Sets the organizational unit that the given session relates to in the given model. This could either be
	 * an Edition or an EditionCollection.
	 *
	 * @param session The session from which to extract the organizational unit.
	 * @param model   The model to set for Thymeleaf resolution.
	 */
	public void setOrganizationInModel(SessionDetailsDTO session, Model model) {
		model.addAttribute("edition",
				(session.getEdition() != null) ? eCache.getOrThrow(session.getEdition().getId()) : null);
		model.addAttribute("ec",
				(session.getEditionCollection() != null)
						? ecCache.getOrThrow(session.getEditionCollection().getId())
						: null);
	}

	/**
	 * Checks whether the given session contains the room with the given id.
	 *
	 * @param  session The session that is to be checked for rooms.
	 * @param  roomId  The id of the room to check for.
	 * @return         Whether the given room is in the given session.
	 */
	public boolean containsRoom(SessionDetailsDTO session, Long roomId) {
		return session.getRooms().stream()
				.anyMatch(r -> Objects.equals(r.getId(), roomId));
	}

	/**
	 * Checks whether the given lab allows the assignment with the given id.
	 *
	 * @param  lab        The lab that needs to be checked.
	 * @param  assignment The assignment that needs to be checked for.
	 * @return            Whether the given assignment is active in the given lab.
	 */
	public boolean containsAssignment(Lab lab, Long assignment) {
		return lab.getAllowedRequests().stream()
				.anyMatch(ar -> Objects.equals(ar.getAssignment(), assignment));
	}

	/**
	 * Checks whether the given lab allows a request with the given assignment and request type.
	 *
	 * @param  lab        The lab that needs to be checked.
	 * @param  assignment The assignment that needs to be checked for.
	 * @param  type       The type that needs to be checked for.
	 * @return            Whether the given assignment and request type combination are allowed in the lab.
	 */
	public boolean containsAllowedRequest(Lab lab, Long assignment, RequestType type) {
		return lab.getAllowedRequests().contains(new AllowedRequest(assignment, type));
	}

	/**
	 * Creates a new lab using the create-DTO filled in when a user submitted the 'Create a new lab' form.
	 * Additional information to this method are the type of organizational unit that the lab related to and
	 * the ID of said organizational unit.
	 *
	 * @param  dto     The create DTO that was submitted by a user through the create page.
	 * @param  ouId    The ID of the organizational unit to create a lab for.
	 * @param  labType The type of lab that is getting created (either a regular lab or a shared lab).
	 * @return         The created lab.
	 */
	@Transactional
	public <D extends Lab> D createLab(LabCreateDTO<D> dto, Long ouId, LabType labType) {
		D lab = dto.apply();

		// Get the assignments and rooms as ID DTOs.
		var assignments = dto.getRequestTypes().keySet().stream()
				.map(a -> new AssignmentIdDTO().id(a))
				.collect(Collectors.toList());
		var rooms = dto.getRooms().stream()
				.map(r -> new RoomIdDTO().id(r))
				.collect(Collectors.toList());

		// Add a session backing up this lab in Labracore.
		switch (labType) {
			case REGULAR:
				lab.setSession(addSingleSession(dto, ouId, assignments, rooms));
				break;
			case SHARED:
				lab.setSession(addSharedSession(dto, ouId, assignments, rooms));
				break;
		}

		// Save the lab and create timeslots for it.
		lab = lr.save(lab);

		if (dto instanceof AbstractSlottedLabCreateDTO && lab instanceof AbstractSlottedLab) {
			tss.createSlotsForLab((AbstractSlottedLabCreateDTO<?>) dto, (AbstractSlottedLab) lab);
		}

		return lab;
	}

	/**
	 * Updates the information of a lab by updating the Queue specific Lab configurations and updating the
	 * session bound to the lab in Labracore.
	 *
	 * @param dto The dto containing information on how to change the lab.
	 * @param lab The lab that is to be edited.
	 */
	@Transactional
	public <D extends Lab> void updateLab(LabPatchDTO<D> dto, D lab) {
		// Get the assignments and rooms as ID DTOs.
		List<AssignmentIdDTO> assignments = null;
		if (dto.getRequestTypes() != null) {
			assignments = dto.getRequestTypes().entrySet().stream()
					.filter(e -> e.getValue() != null && !e.getValue().isEmpty())
					.map(e -> new AssignmentIdDTO().id(e.getKey()))
					.collect(Collectors.toList());
		}

		List<RoomIdDTO> rooms = null;
		if (dto.getRooms() != null) {
			rooms = dto.getRooms().stream()
					.map(r -> new RoomIdDTO().id(r))
					.collect(Collectors.toList());
		}

		// Patch the session on the Labracore side of things.
		sApi.patchSession(lab.getSession(), new SessionPatchDTO()
				.name(dto.getName())
				.start(dto.getSlot().getOpensAt())
				.end(dto.getSlot().getClosesAt())
				.assignments(assignments)
				.rooms(rooms)).block();

		dto.apply(lab);
	}

	/**
	 * Delete a lab by setting the deletedAt to the current date.
	 *
	 * @param lab The lab to delete.
	 */
	@Transactional
	public void deleteLab(Lab lab) {
		lab.setDeletedAt(LocalDateTime.now());
	}

	/**
	 * Adds a regular session in Labracore to represent the Lab. This session is created every time a new lab
	 * gets created, just to back a Queue lab with a core session.
	 *
	 * @param  dto         The DTO used for creating the lab, containing information about the intended
	 *                     session.
	 * @param  editionId   The id of the edition to create this regular lab in.
	 * @param  assignments The assignments that are to be registered for the session.
	 * @param  rooms       The rooms that are to be registered for the session.
	 * @return             The id of the created session.
	 */
	private Long addSingleSession(LabCreateDTO<?> dto, Long editionId,
			List<AssignmentIdDTO> assignments, List<RoomIdDTO> rooms) {
		return sApi.addSingleSession(new SingleSessionCreateDTO()
				.name(dto.getName())
				.description("Queue lab session")
				.start(dto.getSlot().getOpensAt())
				.end(dto.getSlot().getClosesAt())
				.assignments(assignments)
				.rooms(rooms)
				.edition(new EditionIdDTO().id(editionId)))
				.block();
	}

	/**
	 * Adds a shared session in Labracore to represent the Lab. This session is created every time a new lab
	 * gets created, just to back a Queue lab with a core session.
	 *
	 * @param  dto                 The DTO used for creating the lab, containing information about the
	 *                             intended session.
	 * @param  editionCollectionId The id of the edition collection to create this shared lab in.
	 * @param  assignments         The assignments that are to be registered for the session.
	 * @param  rooms               The rooms that are to be registered for the session.
	 * @return                     The id of the created session.
	 */
	private Long addSharedSession(LabCreateDTO<?> dto, Long editionCollectionId,
			List<AssignmentIdDTO> assignments, List<RoomIdDTO> rooms) {
		return sApi.addSharedSession(new SharedSessionCreateDTO()
				.name(dto.getName())
				.description("Queue lab session")
				.start(dto.getSlot().getOpensAt())
				.end(dto.getSlot().getClosesAt())
				.assignments(assignments)
				.rooms(rooms)
				.editionCollection(new EditionCollectionIdDTO().id(editionCollectionId)))
				.block();
	}

}
