/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.StudentGroupControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.queue.cache.AssignmentCacheManager;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.dto.create.RequestCreateDTO;
import nl.tudelft.queue.dto.util.RequestTableFilterDTO;
import nl.tudelft.queue.model.ClosableTimeSlot;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.events.*;
import nl.tudelft.queue.model.labs.ExamLab;
import nl.tudelft.queue.repository.RequestEventRepository;
import nl.tudelft.queue.repository.RequestRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import reactor.core.publisher.Mono;

@Service
public class RequestService {
	@Autowired
	private RequestRepository rr;

	@Autowired
	private RequestEventRepository rer;

	@Autowired
	private QueuePushService qps;

	@Autowired
	private JitsiService js;

	@Autowired
	private WebSocketService wss;

	@Autowired
	private StudentGroupControllerApi sgApi;

	@Autowired
	private AssignmentCacheManager aCache;

	@Autowired
	private PersonCacheManager pCache;

	@Autowired
	private SessionCacheManager sCache;

	/**
	 * Creates a request in the request daabase based on the person that posted the request and the
	 * information the user posted in the form of a {@link RequestCreateDTO}.
	 *
	 * @param dto       The DTO containing user request information.
	 * @param personId  The id of the person that posted the request.
	 * @param sendEvent Whether to send an event to all users that the request was posted (mostly useful in
	 *                  development settings, where creation of a request is not http-request-rooted).
	 */
	@Transactional
	public void createRequest(RequestCreateDTO dto, Long personId, boolean sendEvent) {
		Request request = dto.apply();
		request.setRequester(personId);
		request.setStudentGroup(sgApi
				.getGroupForPersonAndAssignment(personId, dto.getAssignment())
				.onErrorResume(e -> createIndividualStudentGroup(dto, personId))
				.block().getId());

		if (request.getLab().getCommunicationMethod().isOnline()) {
			request.setJitsiRoom(js.createJitsiRoomName(request));
		}

		request = rr.save(request);
		var event = rer.applyAndSave(new RequestCreatedEvent(request));

		if (sendEvent) {
			wss.sendRequestCreated(event);
		}
	}

	/**
	 * Marks a request as taken through the publishing of a {@link RequestTakenEvent}.
	 *
	 * @param request   The request that is to be marked taken.
	 * @param assistant The assistant that took the request.
	 */
	public void takeRequest(Request request, Person assistant) {
		var event = rer.applyAndSave(new RequestTakenEvent(request, assistant.getId()));
		qps.notifyAssistantComing(request, assistant);
		wss.sendRequestTaken(event);
		wss.sendRequestPositionUpdate(request, request.getLab().getQueue());
	}

	/**
	 * Marks a request as forwarded to a specific person.
	 *
	 * @param request            The request to forward.
	 * @param assistant          The assistant that the request is forwarded to.
	 * @param reasonForAssistant The reason the request is forwarded given by the current assistant.
	 */
	public void forwardRequestToPerson(Request request, Person assistant, PersonSummaryDTO forwardedTo,
			String reasonForAssistant) {
		var event = rer.applyAndSave(new RequestForwardedToPersonEvent(
				request, assistant.getId(), forwardedTo.getId(), reasonForAssistant));
		qps.notifyRequestForwarded(request, assistant, forwardedTo);
		wss.sendRequestForwardedToPerson(event);
	}

	/**
	 * Marks a request as forwarded to no one in specific.
	 *
	 * @param request            The request that is to be forwarded.
	 * @param reasonForAssistant The reason the request is forwarded given by the current assistant.
	 */
	public void forwardRequestToAnyone(Request request, Person assistant, String reasonForAssistant) {
		var event = rer
				.applyAndSave(new RequestForwardedToAnyEvent(request, assistant.getId(), reasonForAssistant));
		qps.notifyRequestForwarded(request, assistant, null);
		wss.sendRequestForwardedToAny(event);
	}

	/**
	 * Approves a request by publishing a {@link RequestApprovedEvent}.
	 *
	 * @param request            The request that is to be approved.
	 * @param assistant          The assistant that approved the request.
	 * @param reasonForAssistant The (optional) reason the request is approved given by the current assistant.
	 */
	public void approveRequest(Request request, Long assistant, String reasonForAssistant) {
		var event = rer.applyAndSave(new RequestApprovedEvent(request, assistant, reasonForAssistant));
		wss.sendRequestHandled(event);
	}

	/**
	 * Rejects a request by publishing a {@link RequestRejectedEvent}.
	 *
	 * @param request            The request that is to be rejected.
	 * @param assistant          The assistant that rejected the request.
	 * @param reasonForAssistant The reason the request is rejected given by the current assistant as can be
	 *                           seen by other TAs.
	 * @param reasonForStudent   The reason the request is rejected as can be seen by the student.
	 */
	public void rejectRequest(Request request, Long assistant, String reasonForAssistant,
			String reasonForStudent) {
		var event = rer.applyAndSave(
				new RequestRejectedEvent(request, assistant, reasonForAssistant, reasonForStudent));
		wss.sendRequestHandled(event);
	}

	/**
	 * Marks a request as 'not found' by publishing a {@link StudentNotFoundEvent}.
	 *
	 * @param request   The request that is to be marked.
	 * @param assistant The assistant that marks the request as 'not found'.
	 */
	public void couldNotFindStudent(Request request, Person assistant) {
		var event = rer.applyAndSave(new StudentNotFoundEvent(request, assistant.getId()));
		qps.notifyCouldNotFind(request, assistant);
		wss.sendRequestNotFound(event);
	}

	/**
	 * Revokes a request by publishing a {@link RequestRevokedEvent}.
	 *
	 * @param request The request that is to be revoked.
	 */
	public void revokeRequest(Request request) {
		var event = rer.applyAndSave(new RequestRevokedEvent(request));
		wss.sendRequestRevoked(event);
	}

	/**
	 * Marks a request as "Not Picked" by publishing a {@link RequestNotPickedEvent}.
	 *
	 * @param request The request that was not picked.
	 */
	public void notPickedRequest(Request request) {
		var event = rer.applyAndSave(new RequestNotPickedEvent(request));
		wss.sendRequestNotPicked(event);
	}

	/**
	 * Takes a next request from the given time slot for the given assistant. This method is central to the
	 * working of the Queue from the exam lab overview page. The order of finding a potential request from the
	 * time slot is defined in this method and should be changed here if need be. This method favours
	 * forwarded-to-person requests. After this, a pre-selected student request or a random request is picked.
	 *
	 * @param  assistant The assistant currently requesting a new request to handle.
	 * @param  timeSlot  The time slot for which the assistant wants to get a new request.
	 * @return           The request that was picked to handle next or nothing if none could be found.
	 */
	@Transactional(Transactional.TxType.REQUIRES_NEW)
	public synchronized Optional<Request> takeNextRequestFromTimeSlot(Person assistant,
			ClosableTimeSlot timeSlot) {
		// If the person is already working on a request, no new event should be created.
		Optional<Request> request = rr.findCurrentlyProcessingRequest(assistant, timeSlot);
		if (request.isPresent()) {
			return request;
		}

		// A new request should be found and assigned to the assistant.
		// First comes forwarded requests, then pre-selected student requests, then any request.
		ExamLab lab = (ExamLab) timeSlot.getLab();
		request = rr.findCurrentlyForwardedRequest(assistant, timeSlot)
				.or(() -> pickRandomRequest(lab, rr.findNextExamRequests(timeSlot, assistant)));

		request.ifPresent(r -> takeRequest(r, assistant));

		return request;
	}

	/**
	 * Finds the next available request and marks it as taken as soon as one is found. This method is central
	 * to the working of Queue. The order of finding a potential next request determines priority and all
	 * subsequently called methods that query the Queue in {@link RequestRepository} determine the order in
	 * which requests are handled.
	 *
	 * @param  assistant The assistant currently requesting a new request to handle.
	 * @param  lab       The lab that the assistant is helping with, wanting a request for.
	 * @param  filter    The filter that the assistant is currently applying to the lab.
	 * @return           The request that was picked to handle next or nothing if none could be found.
	 */
	@Transactional(Transactional.TxType.REQUIRES_NEW)
	public synchronized Optional<Request> takeNextRequest(Person assistant, Lab lab,
			RequestTableFilterDTO filter) {
		// If the person is already working on a request, no new event should be created.
		Optional<Request> request = rr.findCurrentlyProcessingRequest(assistant, lab);
		if (request.isPresent()) {
			return request;
		}

		// A new request should be found and assigned to the assistant.
		// First comes forwarded requests, then any other type of request.
		request = rr.findCurrentlyForwardedRequest(assistant, lab)
				.or(() -> findNextRequest(lab, assistant, filter));

		request.ifPresent(r -> takeRequest(r, assistant));

		return request;
	}

	/**
	 * Assigns a specific request to a user. This however can only be done if the user has no processing
	 * requests open.
	 *
	 * @param  assistant The assistant currently requesting to be assigned to a request.
	 * @param  request   The request to which the assistant is assigned.
	 * @return           The request the assistant is currently assigned to.
	 */
	@Transactional(Transactional.TxType.REQUIRES_NEW)
	public synchronized Request pickRequest(Person assistant, Request request) {
		Optional<Request> oldRequests = rr.findCurrentlyProcessingRequest(assistant,
				request.getLab());
		if (oldRequests.isPresent()) {
			return oldRequests.get();
		}
		takeRequest(request, assistant);
		return request;
	}

	/**
	 * Finds the next lab request. This could be a picked request, a timeslot request in a slotted lab or a
	 * normal request in any other lab. This method picks which one it should try to find based on the
	 * settings of the lab.
	 *
	 * @param  lab    The lab to find the next request for.
	 * @param  filter The filter that is currently applied to the requests table.
	 * @return        A fresh request for the assistant to take or an empty Optional if none is found.
	 */
	private Optional<Request> findNextRequest(Lab lab, Person assistant, RequestTableFilterDTO filter) {
		switch (lab.getType()) {
			case SLOTTED:
				return rr.findNextSlotRequest(lab, assistant, filter);
			case EXAM:
				ExamLab examLab = (ExamLab) lab;
				return examLab.getTimeSlots().stream().filter(ClosableTimeSlot::getActive)
						.flatMap(ts -> pickRandomRequest(examLab, rr.findNextExamRequests(ts, assistant))
								.stream())
						.findFirst();
			default:
				return rr.findNextNormalRequest(lab, assistant, filter);
		}
	}

	/**
	 * Picks a random request from a list of requests if a request can be picked. This method favours the
	 * picking of students that were pre-selected by teachers of the course.
	 *
	 * @param  lab      The lab that students were pre-picked in.
	 * @param  requests The request list to pick a request from.
	 * @return          The picked request or none if the list is empty.
	 */
	private Optional<Request> pickRandomRequest(ExamLab lab, List<Request> requests) {
		if (!lab.getPickedStudents().isEmpty() && requests.stream()
				.anyMatch(r -> lab.getPickedStudents().contains(r.getRequester()))) {
			requests = requests.stream()
					.filter(r -> lab.getPickedStudents().contains(r.getRequester()))
					.collect(Collectors.toList());
		}

		if (requests.isEmpty()) {
			return Optional.empty();
		}

		return requests.stream().skip(new Random().nextInt(requests.size())).findFirst();
	}

	/**
	 * Creates an individual student group for one student without a current group so that they may place a
	 * request with that student group.
	 *
	 * TODO: Create a two-in-one function in StudentGroupController to check whether one exists and create a
	 * student group if one does not exist yet.
	 *
	 * @param  dto      The DTO containing user request information.
	 * @param  personId The id of the person that posted the request.
	 * @return          A mono containing the created student group.
	 */
	private Mono<StudentGroupDetailsDTO> createIndividualStudentGroup(RequestCreateDTO dto, Long personId) {
		var person = pCache.getOrThrow(personId);
		var assignment = aCache.getOrThrow(dto.getAssignment());
		var session = sCache.getOrThrow(dto.getLab().getSession());

		return sgApi.addGroup(new StudentGroupCreateDTO()
				.name(person.getDisplayName())
				.capacity(1)
				.addMembersItem(new RoleIdDTO().id(new Id()
						.personId(person.getId())
						.editionId(session.getEdition().getId())))
				.module(new ModuleIdDTO().id(assignment.getModule().getId())))
				.flatMap(id -> sgApi.getStudentGroupsById(List.of(id)).collectList().map(l -> l.get(0)));
	}
}
