/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import static nl.tudelft.labracore.api.dto.RolePersonDetailsDTO.TypeEnum.*;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.http.HttpSession;

import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.SessionControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.cache.*;
import nl.tudelft.queue.dto.util.RequestTableFilterDTO;
import nl.tudelft.queue.dto.view.LabSummaryDTO;
import nl.tudelft.queue.dto.view.RequestViewDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.enums.LabType;
import nl.tudelft.queue.repository.LabRepository;
import nl.tudelft.queue.repository.RequestRepository;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.stereotype.Service;
import org.springframework.ui.Model;

@Service
public class RequestTableService {

	@Autowired
	private HttpSession session;

	@Autowired
	private LabRepository lr;

	@Autowired
	private RequestRepository rr;

	@Autowired
	private EditionControllerApi eApi;

	@Autowired
	private SessionControllerApi sApi;

	@Autowired
	private AssignmentCacheManager aCache;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private EditionRolesCacheManager erCache;

	@Autowired
	private ModuleCacheManager mCache;

	@Autowired
	private PersonCacheManager pCache;

	@Autowired
	private RoomCacheManager rCache;

	@Autowired
	private SessionCacheManager sCache;

	/**
	 * Clears the filter currently stored in the session.
	 *
	 * @param key The key used to store the filter inside the session.
	 */
	public void clearFilter(String key) {
		session.removeAttribute(key);
	}

	/**
	 * Checks the filter DTO for validation errors and stores it in the cache if none are found.
	 *
	 * @param  filter The filter to check and possibly store.
	 * @param  key    The key used to check and store the filter inside the session.
	 * @return        The finalized and validated request table filter.
	 */
	public RequestTableFilterDTO checkAndStoreFilterDTO(RequestTableFilterDTO filter, String key) {
		if (filter == null || filter.hasErrors()) {
			filter = (RequestTableFilterDTO) session.getAttribute(key);
			if (filter == null) {
				filter = new RequestTableFilterDTO();
			}
		} else {
			session.setAttribute(key, filter);
		}

		return filter;
	}

	/**
	 * Gets a mapping from lab ids to the number of requests that are currently in the queue for that lab.
	 *
	 * @param  labs      The labs that we should count over.
	 * @param  assistant The assistant that is requesting this information.
	 * @param  filter    The filter to apply to the request table, given by the user.
	 * @return           The mapping of all lab ids to the number of requests currently still open for that
	 *                   lab.
	 */
	public Map<Long, Long> labRequestCounts(List<Lab> labs, Person assistant, RequestTableFilterDTO filter) {
		return labs.stream().collect(Collectors.toMap(
				Lab::getId,
				l -> {
					if (l.getType() == LabType.EXAM) {
						return rr.countOpenSlotRequests(l, assistant, filter);
						// TODO: count exam lab slots
					} else if (l.getType() == LabType.SLOTTED) {
						return rr.countOpenSlotRequests(l, assistant, filter);
					} else {
						return rr.countNormalRequests(l, assistant, filter);
					}
				}));
	}

	/**
	 * Adds filter specific attributes to the given Model and looks up the set of labs that the currently
	 * authenticated person is involved with based on the active editions they are staff of.
	 *
	 * @param  model The model to fill out for Thymeleaf template resolution.
	 * @param  labs  The labs that are currently active in the request table or {@code null} if the user is
	 *               viewing the request table specifically.
	 * @return       The set of ids of labs that the currently authenticated person is involved with.
	 */
	public List<Lab> addFilterAttributes(Model model, List<Lab> labs) {
		// Fetch the editions currently active and assisted by the user.
		List<EditionDetailsDTO> editions = eCache.get(eApi.getAllEditionsCurrentlyAssistedBy()
				.map(EditionSummaryDTO::getId).collectList().blockOptional().orElse(List.of()));

		// Get all information necessary to display filters.
		// If no list of labs is provided, take all the labs from the editions.
		labs = (labs == null) ? labs(editions) : labs;
		List<SessionDetailsDTO> sessions = sessions(labs);
		List<AssignmentDetailsDTO> assignments = assignments(sessions);
		List<RoomSummaryDTO> rooms = rooms(sessions);
		List<PersonSummaryDTO> assistants = assistants(sessions);

		// Add filter lists to the model for the current request.
		model.addAttribute("editions", editions);
		model.addAttribute("labs", View.convert(labs, LabSummaryDTO.class));
		model.addAttribute("assignments", assignments);
		model.addAttribute("rooms", rooms);
		model.addAttribute("assistants", assistants);

		// Return the list of lab ids for getting requests with.
		return labs;
	}

	/**
	 * Caches information that needs to be fetched about the given list of requests before converting the
	 * entire list of requests into a consumable list of request views. This list of request views can then be
	 * displayed on a page using all the information then available to the Thymeleaf template processor.
	 *
	 * @param  requests The list of requests to process into request views.
	 * @return          The processed list of request views.
	 */
	public List<RequestViewDTO> convertRequestsToView(List<Request> requests) {
		pCache.get(requests.stream()
				.map(Request::getRequester)
				.collect(Collectors.toList()));

		return View.convert(requests, RequestViewDTO.class);
	}

	/**
	 * Caches information that needs to be fetched about the given page of requests before converting the
	 * entire page of requests into a consumable page of request views. This page of request views can then be
	 * displayed on a page using all the information then available to the Thymeleaf template processor.
	 *
	 * @param  requests The page of requests to process into request views.
	 * @return          The processed page of request views.
	 */
	public Page<RequestViewDTO> convertRequestsToView(Page<Request> requests) {
		return new PageImpl<>(convertRequestsToView(requests.getContent()),
				requests.getPageable(), requests.getSize());
	}

	/**
	 * Gets the list of all assistants in a list of editions. This list will be necessary for filtering on a
	 * certain assistant within the Queue requests page.
	 *
	 * @param  editions The list of editions to find assistants in.
	 * @return          The list of all assistants in the given list of course editions.
	 */
	private List<PersonSummaryDTO> assistants(List<SessionDetailsDTO> sessions) {
		return sessions.stream()
				.flatMap(s -> s.getEditions().stream()).distinct()
				.flatMap(e -> erCache.getOrThrow(e.getId()).getRoles().stream())
				.filter(role -> Set.of(TA, HEAD_TA, TEACHER, TEACHER_RO).contains(role.getType()))
				.map(RolePersonDetailsDTO::getPerson)
				.distinct()
				.collect(Collectors.toList());
	}

	/**
	 * Gets the list of all rooms used within the given list of sessions. This information can be used to
	 * display requests and filter requests on the room-specific parameter.
	 *
	 * @param  sessions The list of sessions to find all rooms for.
	 * @return          The list of all rooms used within the given sessions.
	 */
	private List<RoomSummaryDTO> rooms(List<SessionDetailsDTO> sessions) {
		return rCache.get(
				sessions.stream().flatMap(l -> l.getRooms().stream().map(RoomSummaryDTO::getId)).distinct());
	}

	/**
	 * Gets the list of all labs within the given list of editions. This information can be used for
	 * displaying requests and for filtering requests.
	 *
	 * @param  editions The list of editions to find labs by.
	 * @return          The list of all labs within the given editions.
	 */
	private List<Lab> labs(List<EditionDetailsDTO> editions) {
		if (editions.isEmpty()) {
			return List.of();
		}

		var sessions = sApi
				.getActiveSessionsInEditions(
						editions.stream().map(EditionDetailsDTO::getId).collect(Collectors.toList()))
				.collectList().block();
		sCache.register(sessions);

		return lr.findAllBySessions(sessions.stream()
				.map(SessionDetailsDTO::getId)
				.distinct()
				.collect(Collectors.toList()));
	}

	/**
	 * Gets the session details for the given list of labs.
	 *
	 * @param  labs The labs of which the sessions need to be looked up.
	 * @return      The list of all session details gotten from the list of labs.
	 */
	private List<SessionDetailsDTO> sessions(List<Lab> labs) {
		return sCache.get(labs.stream().map(Lab::getSession).distinct().collect(Collectors.toList()));
	}

	/**
	 * Gets the list of all assignments acted upon within the given list of editions. This information can be
	 * used for displaying requests and for filtering requests.
	 *
	 * @param  editions The list of editions to find assignments for.
	 * @return          The list of all assignments within the given editions.
	 */
	private List<AssignmentDetailsDTO> assignments(List<SessionDetailsDTO> sessions) {
		return aCache.get(sessions.stream()
				.flatMap(s -> s.getAssignments().stream())
				.map(AssignmentSummaryDTO::getId).distinct());
	}

}
