/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.when;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.model;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.time.LocalDateTime;
import java.util.List;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.TestQueueApplication;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.EditionRolesCacheManager;
import nl.tudelft.queue.service.PermissionService;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.HttpStatus;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.web.server.ResponseStatusException;

import reactor.core.publisher.Mono;
import test.TestUserDetailsService;

@Transactional
@AutoConfigureMockMvc
@SpringBootTest(classes = TestQueueApplication.class)
public class EditionControllerTest {
	private static final CohortSummaryDTO cohort1 = new CohortSummaryDTO()
			.id(829087322L)
			.name("gdasidjoasdhas");

	private static final CourseSummaryDTO course1 = new CourseSummaryDTO()
			.id(7874234L)
			.code("893LD")
			.name("482nkdskfnk")
			.isArchived(false);

	private static final EditionDetailsDTO edition1 = new EditionDetailsDTO()
			.id(8632L)
			.name("ligdshvhads")
			.cohort(cohort1)
			.course(course1)
			.endDate(LocalDateTime.now().plusYears(1))
			.startDate(LocalDateTime.now())
			.enrollability(EditionDetailsDTO.EnrollabilityEnum.OPEN)
			.isArchived(false);

	private static final EditionRolesCacheManager.RoleHolder emptyRoles = new EditionRolesCacheManager.RoleHolder(
			edition1.getId(), List.of());

	@Autowired
	private MockMvc mvc;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private EditionRolesCacheManager erCache;

	@Autowired
	private EditionControllerApi eApi;

	@MockBean
	private PermissionService ps;

	@Test
	@WithUserDetails("teacher1")
	void getEmptyCourseListContainsEditionPage() throws Exception {
		when(eApi.getEditionsPageActiveOrTaughtBy(eq(TestUserDetailsService.id), any()))
				.thenReturn(Mono.just(new PageEditionDetailsDTO().content(List.of()).totalElements(0L)));
		when(eCache.get(List.of())).thenReturn(List.of());

		mvc.perform(get("/editions"))
				.andExpect(status().isOk())
				.andExpect(model().attributeExists("editions"));
	}

	@Test
	@WithUserDetails("student")
	void getEditionEnrolViewContainsEdition() throws Exception {
		when(ps.canEnrolForEdition(edition1.getId())).thenReturn(true);
		when(eCache.getOrThrow(edition1.getId())).thenReturn(edition1);

		mvc.perform(get("/edition/{id}/enrol", edition1.getId()))
				.andExpect(status().isOk())
				.andExpect(model().attribute("edition", edition1));
	}

	@Test
	@WithUserDetails("student")
	void getEditionEnrolViewGiven404() throws Exception {
		when(ps.canEnrolForEdition(edition1.getId())).thenReturn(true);
		when(eCache.getOrThrow(edition1.getId()))
				.thenThrow(new ResponseStatusException(HttpStatus.NOT_FOUND));

		mvc.perform(get("/edition/{id}/enrol", edition1.getId()))
				.andExpect(status().isNotFound());
	}

	@Test
	@WithUserDetails("student")
	void studentsShouldBeAbleToPostEnrolIfNotYetEnrolled() throws Exception {
		when(ps.canEnrolForEdition(edition1.getId())).thenReturn(true);
		when(eCache.getOrThrow(edition1.getId())).thenReturn(edition1);

		mvc.perform(post("/edition/{id}/enrol", edition1.getId()).with(csrf()))
				.andExpect(status().isOk());
	}

	@Test
	@WithUserDetails("student")
	void studentsShouldNotBeAbleToPostEnrolIfAlreadyEnrolled() throws Exception {
		when(ps.canEnrolForEdition(edition1.getId())).thenReturn(false);
		when(eCache.getOrThrow(edition1.getId())).thenReturn(edition1);

		mvc.perform(post("/edition/{id}/enrol", edition1.getId()).with(csrf()))
				.andExpect(status().isForbidden());
	}

	@Test
	@WithUserDetails("student")
	void studentsShouldBeAbleToSeeEditionInfo() throws Exception {
		when(eCache.getOrThrow(edition1.getId())).thenReturn(edition1);
		when(erCache.getOrThrow(edition1.getId())).thenReturn(emptyRoles);

		mvc.perform(get("/edition/{id}", edition1.getId()))
				.andExpect(status().isOk())
				.andExpect(model().attribute("edition", edition1));
	}

	@Test
	@WithUserDetails("student")
	void getEditionParticipantsViewCallsSearchServiceIfProvided() {

	}
}
