/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.*;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.StudentGroupControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.TestQueueApplication;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.dto.util.RequestTableFilterDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.RegularLab;
import nl.tudelft.queue.repository.LabRepository;
import nl.tudelft.queue.repository.RequestRepository;
import nl.tudelft.queue.service.PermissionService;
import nl.tudelft.queue.service.RequestTableService;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.ArgumentCaptor;
import org.modelmapper.ModelMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.data.domain.*;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.MockHttpServletRequestBuilder;
import org.springframework.ui.Model;

import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import test.TestUserDetailsService;

@AutoConfigureMockMvc
@SpringBootTest(classes = TestQueueApplication.class)
class HistoryControllerTest {

	private static final Long PERSON_ID = TestUserDetailsService.id;
	private static final Long STUDENT_ID = 8684L;
	private static final Long SESSION_ID = 848570L;
	private static final Long EDITION_ID = 48597L;
	private static final Long COURSE_ID = 9541345L;
	private static final Long GROUP_ID = 33859L;
	private static final Long LAB_ID = 48057L;

	private EditionDetailsDTO EDITION_DETAILS;
	private PersonDetailsDTO PERSON_DETAILS;
	private StudentGroupSummaryDTO GROUP_SUMMARY;
	private String path = "";

	private Lab lab;
	private RequestTableFilterDTO filter = new RequestTableFilterDTO();
	private Page<Request> page = new PageImpl<>(List.of());

	@Autowired
	private MockMvc mvc;

	@MockBean
	private RequestTableService service;

	@MockBean
	private PermissionService permissionService;

	@Autowired
	private EditionCacheManager editionCache;

	@Autowired
	private PersonCacheManager personCache;

	@Autowired
	private PersonControllerApi personApi;

	@Autowired
	private StudentGroupControllerApi groupApi;

	@MockBean
	private LabRepository labRepository;

	@MockBean
	private RequestRepository requestRepository;

	@BeforeEach
	void setup() {
		EDITION_DETAILS = new EditionDetailsDTO().id(EDITION_ID)
				.sessions(List.of(new SessionSummaryDTO().id(SESSION_ID)))
				.course(new CourseSummaryDTO().id(COURSE_ID));
		PERSON_DETAILS = new PersonDetailsDTO().id(PERSON_ID)
				.roles(List.of(
						new RoleEditionLayer1DTO().type(RoleEditionLayer1DTO.TypeEnum.STUDENT)
								.edition(new EditionModulesLayer2DTO().id(EDITION_ID)),
						new RoleEditionLayer1DTO().type(RoleEditionLayer1DTO.TypeEnum.TEACHER)));
		GROUP_SUMMARY = new StudentGroupSummaryDTO().id(GROUP_ID);

		lab = RegularLab.builder().id(LAB_ID).build();

		filter.getRequestTypes().add(RequestType.QUESTION);

		when(editionCache.get(any(Stream.class))).thenReturn(List.of(EDITION_DETAILS));

		when(labRepository.findAllBySessions(anyList())).thenReturn(List.of(lab));
		when(requestRepository.findAllPreviousRequests(anyList(), any(RequestTableFilterDTO.class),
				any(Pageable.class))).thenReturn(page);

		when(service.checkAndStoreFilterDTO(any(), anyString())).thenReturn(filter);
		when(service.addFilterAttributes(any(Model.class), anyList())).thenReturn(List.of());
		when(service.convertRequestsToView(any(Page.class))).thenReturn(page);

		when(permissionService.canViewRequests()).thenReturn(false);
	}

	private static class LabList extends ArrayList<Lab> {
	}

	void checkUp() {
		var a = ArgumentCaptor.forClass(Stream.class);
		verify(editionCache).get(a.capture());
		assertThat(a.getValue().collect(Collectors.toList())).isEqualTo(List.of(EDITION_ID));

		verify(labRepository).findAllBySessions(List.of(SESSION_ID));
		verify(requestRepository).findAllPreviousRequests(List.of(GROUP_ID), filter,
				PageRequest.of(0, 25, Sort.by(Sort.Direction.DESC, "createdAt")));

		verify(service).checkAndStoreFilterDTO(null, path);
		var model = ArgumentCaptor.forClass(Model.class);
		var labList = ArgumentCaptor.forClass(LabList.class);
		verify(service).addFilterAttributes(model.capture(), labList.capture());
		assertThat(labList.getValue()).containsExactly(lab);
		verify(service).convertRequestsToView(page);

		verify(permissionService).canViewRequests();
	}

	@Test
	@WithUserDetails("username")
	void getHistoryView() throws Exception {
		path = "/history";
		when(groupApi.getGroupsForPerson(anyLong())).thenReturn(Flux.just(GROUP_SUMMARY));
		when(personApi.getPersonById(anyLong())).thenReturn(Mono.just(PERSON_DETAILS));

		mvc.perform(get(path))
				.andExpect(status().isOk())
				.andExpect(view().name("history/index"));

		verify(groupApi).getGroupsForPerson(PERSON_ID);
		verify(personApi).getPersonById(PERSON_ID);

		checkUp();
	}

	@Test
	@WithUserDetails("username")
	void getStudentHistoryViewVerifiesIsAdminOrTeacher() throws Exception {
		path = "/history/course/" + EDITION_ID + "/student/" + STUDENT_ID;
		when(permissionService.isAdminOrTeacher()).thenReturn(false);

		mvc.perform(get(path))
				.andExpect(status().isForbidden());

		verify(permissionService).isAdminOrTeacher();
	}

	@Test
	@WithUserDetails("username")
	void getStudentHistoryViewAllowsIfIsAdminOrTeacher() throws Exception {
		path = "/history/course/" + EDITION_ID + "/student/" + STUDENT_ID;
		PersonDetailsDTO student = new PersonDetailsDTO().id(STUDENT_ID)
				.displayName("I am Groot")
				.defaultRole(PersonDetailsDTO.DefaultRoleEnum.STUDENT)
				.roles(List.of(
						new RoleEditionLayer1DTO().type(RoleEditionLayer1DTO.TypeEnum.STUDENT)
								.edition(new EditionModulesLayer2DTO().id(EDITION_ID))));
		var summary = new ModelMapper().map(student, PersonSummaryDTO.class);
		when(permissionService.isAdminOrTeacher()).thenReturn(true);
		when(personApi.getPersonById(anyLong())).thenReturn(Mono.just(student));
		when(editionCache.getOrThrow(anyLong())).thenReturn(EDITION_DETAILS);
		when(groupApi.getGroupsForPersonAndCourse(anyLong(), anyLong())).thenReturn(Flux.just(GROUP_SUMMARY));
		when(personCache.getOrThrow(anyLong())).thenReturn(summary);

		mvc.perform(get(path))
				.andExpect(status().isOk())
				.andExpect(model().attribute("edition", EDITION_DETAILS))
				.andExpect(model().attribute("student", summary))
				.andExpect(view().name("history/student"));

		verify(permissionService).isAdminOrTeacher();
		verify(editionCache).getOrThrow(EDITION_ID);
		verify(groupApi).getGroupsForPersonAndCourse(STUDENT_ID, COURSE_ID);
		verify(personApi).getPersonById(STUDENT_ID);
		verify(personCache).getOrThrow(STUDENT_ID);

		checkUp();
	}

	@ParameterizedTest
	@MethodSource(value = "protectedEndpoints")
	void testWithoutUserDetailsIsForbidden(MockHttpServletRequestBuilder request) throws Exception {
		mvc.perform(request.with(csrf()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("http://localhost/login"));
	}

	private static List<MockHttpServletRequestBuilder> protectedEndpoints() {
		return List.of(
				get("/history"),
				get("/history/course/1/student/1"));
	}

}
