/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.RoleControllerApi;
import nl.tudelft.labracore.api.dto.PersonDetailsDTO;
import nl.tudelft.labracore.api.dto.PersonSummaryDTO;
import nl.tudelft.queue.TestQueueApplication;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.model.Feedback;
import nl.tudelft.queue.repository.FeedbackRepository;
import nl.tudelft.queue.service.PermissionService;

import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.web.servlet.MockMvc;

import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import test.TestUserDetailsService;

@AutoConfigureMockMvc
@SpringBootTest(classes = TestQueueApplication.class)
class HomeControllerTest {

	private static final Long PERSON_ID = TestUserDetailsService.id;

	@Autowired
	private MockMvc mvc;

	@MockBean
	private PermissionService permission;

	@Autowired
	private PersonControllerApi personApi;

	@Autowired
	private PersonCacheManager personCache;

	@Autowired
	private RoleControllerApi roleApi;

	@Autowired
	private EditionControllerApi editionApi;

	@MockBean
	private FeedbackRepository feedbackRepository;

	@Autowired
	private EditionCacheManager editionCache;

	@Test
	void page() {
	}

	@Test
	void privacyStatement() throws Exception {
		mvc.perform(get("/privacy"))
				.andExpect(status().isOk())
				.andExpect(view().name("home/privacy"));
	}

	@Test
	void about() throws Exception {
		mvc.perform(get("/about"))
				.andExpect(status().isOk())
				.andExpect(view().name("home/about"));
	}

	@Test
	void indexChecksAuth() throws Exception {
		mvc.perform(get("/"))
				.andExpect(status().isOk())
				.andExpect(view().name("home/index"));
	}

	@Test
	@WithUserDetails("username")
	void index() throws Exception {
		PersonDetailsDTO person = new PersonDetailsDTO().id(PERSON_ID)
				.displayName("Display Name");
		when(personApi.getPersonById(anyLong())).thenReturn(Mono.just(person));
		when(roleApi.getPersonRolesInActiveEditions(anyLong())).thenReturn(Flux.empty());
		when(editionCache.get(any(Stream.class))).thenReturn(List.of());

		mvc.perform(get("/"))
				.andExpect(status().isOk())
				.andExpect(model().attribute("user", person))
				.andExpect(model().attribute("roles", List.of()))
				.andExpect(model().attribute("editions", Map.of()))
				.andExpect(model().attributeDoesNotExist("runningEditions"))
				.andExpect(view().name("home/dashboard"));

		verify(personApi).getPersonById(PERSON_ID);
		verify(roleApi).getPersonRolesInActiveEditions(PERSON_ID);
		var a = ArgumentCaptor.forClass(Stream.class);
		verify(editionCache).get(a.capture());
		assertThat(a.getValue().collect(Collectors.toList())).isEqualTo(List.of());
	}

	@Test
	@WithUserDetails("username")
	void indexRecognizesAdmin() throws Exception {
		PersonDetailsDTO person = new PersonDetailsDTO().id(PERSON_ID)
				.defaultRole(PersonDetailsDTO.DefaultRoleEnum.ADMIN);
		when(personApi.getPersonById(anyLong())).thenReturn(Mono.just(person));
		when(roleApi.getPersonRolesInActiveEditions(anyLong())).thenReturn(Flux.empty());
		when(editionCache.get(any(Stream.class))).thenReturn(List.of());
		when(editionApi.getAllEditionsActiveAtDate(any(LocalDateTime.class)))
				.thenReturn(Flux.empty());

		mvc.perform(get("/"))
				.andExpect(status().isOk())
				.andExpect(model().attribute("user", person))
				.andExpect(model().attribute("roles", List.of()))
				.andExpect(model().attribute("editions", Map.of()))
				.andExpect(model().attribute("runningEditions", List.of()))
				.andExpect(view().name("home/dashboard"));

		verify(personApi).getPersonById(PERSON_ID);
		verify(roleApi).getPersonRolesInActiveEditions(PERSON_ID);
		var a = ArgumentCaptor.forClass(Stream.class);
		verify(editionCache).get(a.capture());
		assertThat(a.getValue().collect(Collectors.toList())).isEqualTo(List.of());
	}

	@Test
	void feedbackNeedsAuthentication() throws Exception {
		mvc.perform(get("/feedback"))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("http://localhost/login"));
	}

	@Test
	@WithUserDetails("username")
	void feedbackVerifiesCanViewFeedback() throws Exception {
		when(permission.canViewOwnFeedback()).thenReturn(false);

		mvc.perform(get("/feedback?page=1&size=1"))
				.andExpect(status().isForbidden());

		verify(permission, times(2)).canViewOwnFeedback();
		verify(personCache, never()).getOrThrow(anyLong());
		verify(feedbackRepository, never()).findByAssistant(anyLong(), any(Pageable.class));
	}

	@Test
	@WithUserDetails("username")
	void feedbackAllowsIfCanViewFeedback() throws Exception {
		PersonSummaryDTO person = new PersonSummaryDTO().id(PERSON_ID).displayName("Display Name");
		Page<Feedback> feedback = new PageImpl<>(List.of());

		when(permission.canViewOwnFeedback()).thenReturn(true);
		when(personCache.getOrThrow(anyLong())).thenReturn(person);
		when(feedbackRepository.findByAssistant(anyLong(), any(Pageable.class)))
				.thenReturn(feedback);

		mvc.perform(get("/feedback?page=1&size=1"))
				.andExpect(status().isOk())
				.andExpect(model().attribute("assistant", person))
				.andExpect(model().attribute("feedback", feedback))
				.andExpect(view().name("home/feedback"));

		verify(feedbackRepository).findByAssistant(PERSON_ID, PageRequest.of(1, 1));
		verify(permission, times(2)).canViewOwnFeedback();
		verify(personCache).getOrThrow(PERSON_ID);
	}

	@Test
	void specificFeedbackNeedsAuthentication() throws Exception {
		mvc.perform(get("/feedback/1"))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("http://localhost/login"));
	}

	@Test
	@WithUserDetails("username")
	void specificFeedbackVerifiesCanViewFeedback() throws Exception {
		when(permission.canViewFeedback(anyLong())).thenReturn(false);

		mvc.perform(get("/feedback/{id}?page=1&size=1", PERSON_ID))
				.andExpect(status().isForbidden());

		verify(permission).canViewFeedback(PERSON_ID);
		verify(personCache, never()).getOrThrow(anyLong());
		verify(feedbackRepository, never()).findByAssistant(anyLong(), any(Pageable.class));
	}

	@Test
	@WithUserDetails("username")
	void specificFeedbackAllowsIfCanViewFeedback() throws Exception {
		Long id = 495L;
		PersonSummaryDTO person = new PersonSummaryDTO().id(id).displayName("Display Name");
		Page<Feedback> feedback = new PageImpl<>(List.of());

		when(permission.canViewFeedback(anyLong())).thenReturn(true);
		when(personCache.getOrThrow(anyLong())).thenReturn(person);
		when(feedbackRepository.findByAssistant(anyLong(), any(Pageable.class)))
				.thenReturn(feedback);

		mvc.perform(get("/feedback/{id}?page=1&size=1", id))
				.andExpect(status().isOk())
				.andExpect(model().attribute("assistant", person))
				.andExpect(model().attribute("feedback", feedback))
				.andExpect(view().name("home/feedback"));

		verify(feedbackRepository).findByAssistant(id, PageRequest.of(1, 1));
		verify(permission).canViewFeedback(id);
		verify(personCache).getOrThrow(id);
	}

}
