/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.redirectedUrl;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Set;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.TestQueueApplication;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.LabType;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.RegularLab;
import nl.tudelft.queue.model.labs.SlottedLab;
import nl.tudelft.queue.repository.LabRepository;
import nl.tudelft.queue.service.LabService;
import nl.tudelft.queue.service.PermissionService;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.web.servlet.MockMvc;

@AutoConfigureMockMvc
@SpringBootTest(classes = TestQueueApplication.class)
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
class LabControllerTest {

	@Autowired
	private MockMvc mockMvc;

	@Autowired
	private LabRepository labRepository;

	@MockBean
	private PermissionService ps;

	@SpyBean
	private LabService labService;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private SessionCacheManager sCache;

	private EditionSummaryDTO editionSummaryDTO;

	private static final EditionDetailsDTO edition1 = new EditionDetailsDTO()
			.id(8632L)
			.name("ligdshvhads")
			.endDate(LocalDateTime.now().plusYears(1))
			.startDate(LocalDateTime.now())
			.enrollability(EditionDetailsDTO.EnrollabilityEnum.OPEN)
			.isArchived(false)
			.course(new CourseSummaryDTO().code("test"));

	private Lab lab;
	private SessionDetailsDTO session1;

	@BeforeEach
	void setup() {
		editionSummaryDTO = new EditionSummaryDTO().id(45241L);
		session1 = new SessionDetailsDTO()
				.rooms(List.of(new RoomSummaryDTO().id(76832L)))
				.edition(editionSummaryDTO).id(12012L).editions(List.of(editionSummaryDTO))
				.start(LocalDateTime.MIN).end(LocalDateTime.MAX);

		lab = labRepository.save(RegularLab.builder()
				.communicationMethod(CommunicationMethod.JITSI_MEET)
				.session(12012L)
				.allowedRequests(Set.of(new AllowedRequest(82L, RequestType.QUESTION)))
				.type(LabType.REGULAR)
				.modules(Set.of(1L))
				.build());
	}

	@Test
	@WithUserDetails("student")
	@Transactional
	void redirectToEnrollPageWhenNotEnrolled() throws Exception {
		when(ps.canViewLab(lab)).thenReturn(false);
		when(sCache.getOrThrow(lab.getSession())).thenReturn(session1);

		mockMvc.perform(get("/lab/{id}", lab.getId()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/edition/" + session1.getEdition().getId() + "/enrol"));
	}

	@Test
	@WithUserDetails("admin")
	@Transactional
	void deleteLab() throws Exception {
		when(ps.canManageLab(any())).thenReturn(true);
		doNothing().when(labService).deleteLab(isA(Lab.class));
		when(sCache.getOrThrow(any())).thenReturn(session1);

		mockMvc.perform(post("/lab/{id}/delete", lab.getId()).with(csrf()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrlTemplate("/edition/{id}/labs", editionSummaryDTO.getId()));

		verify(labService, times(1)).deleteLab(any(Lab.class));
	}

	@Test
	@WithUserDetails("admin")
	@Transactional
	void getSlottedLabTest() throws Exception {
		lab = labRepository.save(SlottedLab.builder()
				.communicationMethod(CommunicationMethod.JITSI_MEET)
				.session(12012L)
				.allowedRequests(Set.of(new AllowedRequest(82L, RequestType.QUESTION)))
				.type(LabType.SLOTTED)
				.modules(Set.of(1L))
				.build());
		when(ps.canViewLab(lab)).thenReturn(true);
		doReturn(session1).when(sCache).getOrThrow(any());
		when(eCache.getOrThrow(any())).thenReturn(edition1);
		//		doCallRealMethod().when(labService).setOrganizationInModel(isA(SessionDetailsDTO.class),
		//				any());

		mockMvc.perform(get("/lab/" + lab.getId())).andExpect(status().is2xxSuccessful());
	}
}
