/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

import java.util.*;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.TestQueueApplication;
import nl.tudelft.queue.cache.*;
import nl.tudelft.queue.dto.util.RequestTableFilterDTO;
import nl.tudelft.queue.dto.view.RequestViewDTO;
import nl.tudelft.queue.dto.view.RequestWithEventsViewDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.Request;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.embeddables.RequestEventInfo;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.LabType;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.RegularLab;
import nl.tudelft.queue.repository.LabRepository;
import nl.tudelft.queue.repository.RequestRepository;
import nl.tudelft.queue.service.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.ArgumentCaptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.security.test.context.support.WithUserDetails;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.MockHttpServletRequestBuilder;
import org.springframework.ui.Model;

import reactor.core.publisher.Mono;
import test.TestUserDetailsService;

@AutoConfigureMockMvc
@SpringBootTest(classes = TestQueueApplication.class)
@Transactional
class RequestControllerTest {
	private static final Long SESSION_ID = 53643L;
	private static final Long ASSIGNMENT_ID = 185684L;
	private static final Long MODULE_ID = 64896L;
	private static final Long ROOM_ID = 876898796L;
	private static final Long REQUESTER_ID = 27856L;
	private static final Long GROUP_ID = 7654168L;
	private static final Long EDITION_ID = 7654168L;

	private Lab lab;
	private Request request;
	private RequestTableFilterDTO filter;

	@Autowired
	private MockMvc mvc;

	@MockBean
	private PermissionService permissionService;

	@MockBean
	private RequestService service;

	@MockBean
	private LabService labService;

	@MockBean
	private RequestTableService requestTableService;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private SessionCacheManager sCache;

	@Autowired
	private PersonCacheManager pCache;

	@Autowired
	private StudentGroupCacheManager gCache;

	@Autowired
	private AssignmentCacheManager aCache;

	@Autowired
	private RoomCacheManager rCache;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private EditionRolesCacheManager erCache;

	@Autowired
	private PersonControllerApi personApi;

	private Lab getLab() {
		return labRepository.findById(lab.getId()).orElseThrow();
	}

	private Request getRequest() {
		return requestRepository.findById(request.getId()).get();
	}

	private RequestViewDTO getRequestViewDTO() {
		RequestViewDTO dto = View.convert(getRequest(), RequestViewDTO.class);
		dto.setAssignment(new AssignmentDetailsDTO().id(ASSIGNMENT_ID).name("Assignment 1"));
		dto.setRequester(new PersonSummaryDTO().displayName("Requester Person").username("requester"));
		dto.setStudentGroup(new StudentGroupDetailsDTO().memberUsernames(List.of("requester")));
		dto.setSession(new SessionDetailsDTO().edition(new EditionSummaryDTO().name("Edition This")));
		dto.setRoom(new RoomSummaryDTO().name("Room That"));
		return dto;
	}

	@BeforeEach
	void setUp() {
		lab = labRepository.save(RegularLab.builder()
				.type(LabType.REGULAR)
				.communicationMethod(CommunicationMethod.JITSI_MEET)
				.session(SESSION_ID)
				.allowedRequests(Set.of(new AllowedRequest(ASSIGNMENT_ID, RequestType.QUESTION)))
				.modules(Set.of(MODULE_ID))
				.build());
		request = requestRepository.save(Request.builder()
				.lab(lab)
				.requestType(RequestType.QUESTION)
				.assignment(ASSIGNMENT_ID)
				.room(ROOM_ID)
				.requester(REQUESTER_ID)
				.studentGroup(GROUP_ID)
				.eventInfo(new RequestEventInfo())
				.build());

		filter = new RequestTableFilterDTO();
		filter.getRequestTypes().add(RequestType.QUESTION);
	}

	@Test
	@WithUserDetails("username")
	void getNextRequestVerifiesCanTakeRequest() throws Exception {
		when(permissionService.canTakeRequest(lab.getId())).thenReturn(false);

		mvc.perform(get("/requests/next/{lab}", lab.getId()))
				.andExpect(status().isForbidden());

		verify(permissionService).canTakeRequest(lab.getId());
	}

	@Test
	@WithUserDetails("username")
	void getNextRequestRedirectsToExistingRequest() throws Exception {
		when(permissionService.canTakeRequest(anyLong())).thenReturn(true);
		when(requestTableService.checkAndStoreFilterDTO(any(), anyString()))
				.thenReturn(filter);
		when(service.takeNextRequest(any(Person.class), any(Lab.class),
				any(RequestTableFilterDTO.class)))
						.thenReturn(Optional.of(request));

		mvc.perform(get("/requests/next/{lab}", lab.getId()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/request/" + request.getId()));

		verify(permissionService).canTakeRequest(lab.getId());
		verify(requestTableService).checkAndStoreFilterDTO(null, "/requests");
		verify(service).takeNextRequest(TestUserDetailsService.person,
				getLab(), filter);
	}

	@Test
	@WithUserDetails("username")
	void getNextRequestRedirectsToAllIfNoRequest() throws Exception {
		when(permissionService.canTakeRequest(anyLong())).thenReturn(true);
		when(requestTableService.checkAndStoreFilterDTO(any(), anyString()))
				.thenReturn(filter);
		when(service.takeNextRequest(any(Person.class), any(Lab.class),
				any(RequestTableFilterDTO.class)))
						.thenReturn(Optional.empty());

		mvc.perform(get("/requests/next/{lab}", lab.getId()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/requests"));

		verify(permissionService).canTakeRequest(lab.getId());
		verify(requestTableService).checkAndStoreFilterDTO(null, "/requests");
		verify(service).takeNextRequest(TestUserDetailsService.person,
				getLab(), filter);
	}

	@Test
	@WithUserDetails("username")
	void getRequestTableViewVerifiesCanViewRequests() throws Exception {
		when(permissionService.canViewRequests()).thenReturn(false);

		mvc.perform(get("/requests"))
				.andExpect(status().isForbidden());

		verify(permissionService, times(2)).canViewRequests(); // once to authorise and once in layout.html
	}

	@Test
	@WithUserDetails("username")
	void getRequestTableViewAllowsIfCanViewRequests() throws Exception {
		lab = getLab();
		Map<Long, Long> map = Map.of(lab.getId(), 1L);
		Page<RequestViewDTO> requestPage = new PageImpl<>(List.of(getRequestViewDTO()));

		when(permissionService.canViewRequests()).thenReturn(true);
		when(requestTableService.checkAndStoreFilterDTO(any(), anyString())).thenReturn(filter);
		when(requestTableService.addFilterAttributes(any(Model.class), any()))
				.thenReturn(List.of(lab));
		when(requestTableService.convertRequestsToView(any(Page.class)))
				.thenReturn(requestPage);
		when(requestTableService.labRequestCounts(anyList(), any(Person.class),
				any(RequestTableFilterDTO.class)))
						.thenReturn(map);

		mvc.perform(get("/requests"))
				.andExpect(status().isOk())
				.andExpect(model().attribute("page", "requests"))
				.andExpect(model().attribute("filter", filter))
				.andExpect(model().attribute("requests", requestPage))
				.andExpect(model().attribute("requestCounts", map));

		verify(permissionService, times(2)).canViewRequests(); // once to authorise and once in layout.html
		verify(requestTableService).checkAndStoreFilterDTO(null, "/requests");
		verify(requestTableService).addFilterAttributes(any(Model.class), eq(null));
		verify(requestTableService).labRequestCounts(List.of(lab), TestUserDetailsService.person, filter);

		var page = ArgumentCaptor.forClass(Page.class);
		verify(requestTableService).convertRequestsToView(page.capture());
		assertThat(page.getValue().toList()).containsExactly(getRequest());
	}

	@Test
	@WithUserDetails("username")
	void updateRequestInfoVerifiesCanUpdateRequest() throws Exception {
		when(permissionService.canUpdateRequest(anyLong())).thenReturn(false);

		mvc.perform(post("/request/{request}/update-request-info", getRequest().getId()).with(csrf()))
				.andExpect(status().isForbidden());

		verify(permissionService).canUpdateRequest(getRequest().getId());
	}

	@Test
	@WithUserDetails("username")
	void updateRequestInfoAllowsIfCanUpdateRequest() throws Exception {
		assertThat(request.getRoom()).isEqualTo(ROOM_ID);
		Long roomId = ROOM_ID + 7L;

		when(permissionService.canUpdateRequest(anyLong())).thenReturn(true);
		when(sCache.getOrThrow(anyLong()))
				.thenReturn(new SessionDetailsDTO().rooms(List.of(new RoomSummaryDTO().id(roomId))));

		mvc.perform(post("/request/{request}/update-request-info", getRequest().getId()).with(csrf())
				.param("room", roomId.toString()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/lab/" + lab.getId()));

		verify(permissionService).canUpdateRequest(getRequest().getId());
		verify(sCache).getOrThrow(SESSION_ID);

		assertThat(getRequest().getRoom()).isEqualTo(roomId);
	}

	@Test
	@WithUserDetails("username")
	void getRequestViewVerifiesCanViewRequest() throws Exception {
		when(permissionService.canViewRequest(anyLong())).thenReturn(false);

		mvc.perform(get("/request/{request}", request.getId()))
				.andExpect(status().isForbidden());

		verify(permissionService).canViewRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	@DirtiesContext(methodMode = DirtiesContext.MethodMode.BEFORE_METHOD)
	void getRequestViewAllowsIfCanViewRequest() throws Exception {
		when(permissionService.canViewRequest(anyLong())).thenReturn(true);
		doAnswer(invocation -> {
			Model m = invocation.getArgument(1);

			m.addAttribute("edition",
					new EditionDetailsDTO()
							.id(EDITION_ID)
							.course(new CourseSummaryDTO().code("C123")));
			m.addAttribute("ec", null);

			return null;
		}).when(labService).setOrganizationInModel(any(Lab.class), any(Model.class));

		when(pCache.getOrThrow(anyLong()))
				.thenReturn(new PersonSummaryDTO().id(REQUESTER_ID).displayName("Requester"));
		when(sCache.getOrThrow(anyLong()))
				.thenReturn(new SessionDetailsDTO().id(SESSION_ID));
		when(gCache.getOrThrow(anyLong()))
				.thenReturn(new StudentGroupDetailsDTO().id(GROUP_ID).name("Group Thing").members(List.of()));
		when(aCache.getOrThrow(anyLong()))
				.thenReturn(new AssignmentDetailsDTO().id(ASSIGNMENT_ID).name("Assignment 1"));
		when(rCache.getOrThrow(anyLong())).thenReturn(new RoomSummaryDTO().id(ROOM_ID).name("Roomie"));

		mvc.perform(get("/request/{request}", request.getId()))
				.andExpect(status().isOk())
				.andExpect(view().name("request/view"))
				.andExpect(model().attribute("request",
						View.convert(getRequest(), RequestWithEventsViewDTO.class)))
				.andExpect(model().attribute("prevRequests", List.of()));

		verify(permissionService).canViewRequest(request.getId());
		verify(labService).setOrganizationInModel(eq(getLab()), any(Model.class));

		// in both super and subclass
		verify(pCache, times(2)).getOrThrow(REQUESTER_ID);
		verify(sCache, times(2)).getOrThrow(SESSION_ID);
		verify(gCache, times(2)).getOrThrow(GROUP_ID);
		verify(aCache, times(2)).getOrThrow(ASSIGNMENT_ID);
		verify(rCache, times(2)).getOrThrow(ROOM_ID);
	}

	@Test
	@WithUserDetails("username")
	@DirtiesContext(methodMode = DirtiesContext.MethodMode.BEFORE_METHOD)
	void getRequestViewFetchesExistingAssistant() throws Exception {
		Long assistantId = 3285468L;
		PersonDetailsDTO assistant = new PersonDetailsDTO().id(assistantId);
		getRequest().setEventInfo(RequestEventInfo.builder().assignedTo(assistantId).build());

		when(permissionService.canViewRequest(anyLong())).thenReturn(true);
		when(personApi.getPersonById(anyLong())).thenReturn(Mono.just(assistant));
		doAnswer(invocation -> {
			Model m = invocation.getArgument(1);

			m.addAttribute("edition",
					new EditionDetailsDTO()
							.id(EDITION_ID)
							.course(new CourseSummaryDTO().code("C123")));
			m.addAttribute("ec", null);

			return null;
		}).when(labService).setOrganizationInModel(any(Lab.class), any(Model.class));

		when(pCache.getOrThrow(anyLong()))
				.thenReturn(new PersonSummaryDTO().id(REQUESTER_ID).displayName("Requester"));
		when(sCache.getOrThrow(anyLong()))
				.thenReturn(new SessionDetailsDTO().id(SESSION_ID));
		when(gCache.getOrThrow(anyLong()))
				.thenReturn(new StudentGroupDetailsDTO().id(GROUP_ID).name("Group Thing").members(List.of()));
		when(aCache.getOrThrow(anyLong()))
				.thenReturn(new AssignmentDetailsDTO().id(ASSIGNMENT_ID).name("Assignment 1"));
		when(rCache.getOrThrow(anyLong())).thenReturn(new RoomSummaryDTO().id(ROOM_ID).name("Roomie"));

		mvc.perform(get("/request/{request}", request.getId()))
				.andExpect(status().isOk())
				.andExpect(view().name("request/view"))
				.andExpect(model().attribute("request",
						View.convert(getRequest(), RequestWithEventsViewDTO.class)))
				.andExpect(model().attribute("prevRequests", List.of()))
				.andExpect(model().attribute("assistant", assistant));

		verify(permissionService).canViewRequest(request.getId());
		verify(personApi).getPersonById(assistantId);
		verify(labService).setOrganizationInModel(eq(getLab()), any(Model.class));

		// in both super and subclass
		verify(pCache, times(2)).getOrThrow(REQUESTER_ID);
		verify(sCache, times(2)).getOrThrow(SESSION_ID);
		verify(gCache, times(2)).getOrThrow(GROUP_ID);
		verify(aCache, times(2)).getOrThrow(ASSIGNMENT_ID);
		verify(rCache, times(2)).getOrThrow(ROOM_ID);
	}

	@Test
	@WithUserDetails("username")
	void getRequestApproveViewVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(get("/request/{request}/approve", request.getId()))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void getRequestApproveViewAllowsIfCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(labService).setOrganizationInModel(any(Lab.class), any(Model.class));

		mvc.perform(get("/request/{request}/approve", request.getId()))
				.andExpect(status().isOk())
				.andExpect(view().name("request/approve"))
				.andExpect(model().attribute("request", getRequest()));

		verify(permissionService).canFinishRequest(request.getId());
		verify(labService).setOrganizationInModel(eq(getLab()), any(Model.class));
	}

	@Test
	@WithUserDetails("username")
	void approveRequestVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(post("/request/{request}/approve", request.getId()).with(csrf())
				.param("reasonForAssistant", ""))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void approveRequestAllowsIfCanFinishRequest() throws Exception {
		String reason = "The answer was, inderdeed, 42.";
		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(service).approveRequest(any(Request.class), anyLong(), anyString());

		mvc.perform(post("/request/{request}/approve", request.getId()).with(csrf())
				.param("reasonForAssistant", reason))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/requests"))
				.andExpect(flash().attribute("message", "Request #" + request.getId() + " approved"));

		verify(permissionService).canFinishRequest(request.getId());
		verify(service).approveRequest(getRequest(), TestUserDetailsService.id, reason);
	}

	@Test
	@WithUserDetails("username")
	void getRequestRejectViewVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(get("/request/{request}/reject", request.getId()))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void getRequestRejectViewAllowsIfCanFinishRequest() throws Exception {
		Request req = getRequest();

		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(labService).setOrganizationInModel(any(Lab.class), any(Model.class));

		mvc.perform(get("/request/{request}/reject", request.getId()))
				.andExpect(status().isOk())
				.andExpect(view().name("request/reject"))
				.andExpect(model().attribute("request", req));

		verify(labService).setOrganizationInModel(eq(req.getLab()), any(Model.class));
		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void rejectRequestVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(post("/request/{request}/reject", request.getId()).with(csrf())
				.param("reasonForAssistant", "")
				.param("reasonForStudent", ""))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void rejectRequestAllowsIfCanFinishRequest() throws Exception {
		String reasonForAssistant = "was very bad";
		String reasonForStudent = "needs some more work";

		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(service).rejectRequest(any(Request.class), anyLong(), anyString(), anyString());

		mvc.perform(post("/request/{request}/reject", request.getId()).with(csrf())
				.param("reasonForAssistant", reasonForAssistant)
				.param("reasonForStudent", reasonForStudent))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/requests"))
				.andExpect(flash().attribute("message", "Request #" + request.getId() + " rejected"));

		verify(service).rejectRequest(getRequest(), TestUserDetailsService.id, reasonForAssistant,
				reasonForStudent);
		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void getRequestForwardViewVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(get("/request/{request}/forward", request.getId()))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void getRequestForwardViewAllowsIfCanFinishRequest() throws Exception {
		Long assistantId = 694L;
		PersonSummaryDTO assistant = new PersonSummaryDTO().id(assistantId);

		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(labService).setOrganizationInModel(any(Lab.class), any(Model.class));
		when(sCache.getOrThrow(anyLong())).thenReturn(new SessionDetailsDTO()
				.id(SESSION_ID)
				.editions(List.of(new EditionSummaryDTO().id(EDITION_ID))));
		when(eCache.getOrThrow(anyLong())).thenReturn(new EditionDetailsDTO().id(EDITION_ID));
		when(erCache.getOrThrow(anyLong())).thenReturn(new EditionRolesCacheManager.RoleHolder(EDITION_ID,
				List.of(new RolePersonDetailsDTO().type(RolePersonDetailsDTO.TypeEnum.STUDENT),
						new RolePersonDetailsDTO().id(new Id().personId(assistantId).editionId(EDITION_ID))
								.type(RolePersonDetailsDTO.TypeEnum.TA).person(assistant),
						new RolePersonDetailsDTO()
								.id(new Id().personId(TestUserDetailsService.id).editionId(EDITION_ID))
								.type(RolePersonDetailsDTO.TypeEnum.HEAD_TA)
								.person(new PersonSummaryDTO().id(TestUserDetailsService.id)))));

		mvc.perform(get("/request/{request}/forward", request.getId()))
				.andExpect(status().isOk())
				.andExpect(view().name("request/forward"))
				.andExpect(model().attribute("request", getRequest()))
				.andExpect(model().attribute("assistants", List.of(assistant)));

		verify(permissionService).canFinishRequest(request.getId());
		verify(labService).setOrganizationInModel(eq(getLab()), any(Model.class));
		verify(sCache).getOrThrow(SESSION_ID);
		verify(eCache).getOrThrow(EDITION_ID);
	}

	@Test
	@WithUserDetails("username")
	void forwardRequestVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(post("/request/{request}/forward", request.getId()).with(csrf())
				.param("assistant", String.valueOf(-1L))
				.param("reasonForAssistant", ""))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void forwardRequestAllowsIfCanFinishRequest() throws Exception {
		String reasonForAssistant = "don't like this kid";
		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(service).forwardRequestToAnyone(any(Request.class), any(Person.class), anyString());

		mvc.perform(post("/request/{request}/forward", request.getId()).with(csrf())
				.param("assistant", String.valueOf(-1L))
				.param("reasonForAssistant", reasonForAssistant))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/requests"))
				.andExpect(flash().attribute("message", "Request #" + request.getId() + " forwarded"));

		verify(service).forwardRequestToAnyone(getRequest(), TestUserDetailsService.person,
				reasonForAssistant);
		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void forwardRequestRegistersForwardedTo() throws Exception {
		Long forwarderId = 485687L;
		PersonSummaryDTO forwarded = new PersonSummaryDTO().id(forwarderId);
		String reasonForAssistant = "don't like this kid";
		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(service).forwardRequestToPerson(any(Request.class), any(Person.class),
				any(PersonSummaryDTO.class), anyString());
		when(pCache.getOrThrow(anyLong())).thenReturn(forwarded);

		mvc.perform(post("/request/{request}/forward", request.getId()).with(csrf())
				.param("assistant", String.valueOf(forwarderId))
				.param("reasonForAssistant", reasonForAssistant))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/requests"))
				.andExpect(flash().attribute("message", "Request #" + request.getId() + " forwarded"));

		verify(service).forwardRequestToPerson(getRequest(), TestUserDetailsService.person, forwarded,
				reasonForAssistant);
		verify(pCache).getOrThrow(forwarderId);
		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void couldNotFindStudentVerifiesCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(false);

		mvc.perform(get("/request/{request}/not-found", request.getId()))
				.andExpect(status().isForbidden());

		verify(permissionService).canFinishRequest(request.getId());
	}

	@Test
	@WithUserDetails("username")
	void couldNotFindStudentAllowsIfCanFinishRequest() throws Exception {
		when(permissionService.canFinishRequest(anyLong())).thenReturn(true);
		doNothing().when(service).couldNotFindStudent(any(Request.class), any(Person.class));

		mvc.perform(get("/request/{request}/not-found", request.getId()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("/requests"));

		verify(service).couldNotFindStudent(getRequest(), TestUserDetailsService.person);
		verify(permissionService).canFinishRequest(request.getId());
	}

	@ParameterizedTest
	@MethodSource(value = "protectedEndpoints")
	void testWithoutUserDetailsIsForbidden(MockHttpServletRequestBuilder request) throws Exception {
		mvc.perform(request.with(csrf()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrl("http://localhost/login"));
	}

	private static List<MockHttpServletRequestBuilder> protectedEndpoints() {
		return List.of(
				get("/requests/next/1"),
				get("/requests"),
				post("/request/1/update-request-info"),
				get("/request/1"),
				get("/request/1/approve"),
				post("/request/1/approve"),
				get("/request/1/reject"),
				post("/request/1/reject"),
				get("/request/1/not-found"));
	}

	@Test
	@WithUserDetails("username")
	void pickRequest() throws Exception {
		when(permissionService.canPickRequest(any())).thenReturn(true);
		when(service.pickRequest(any(Person.class), any(Request.class)))
				.thenReturn(request);

		mvc.perform(get("/request/{request}/pick", request.getId()).with(csrf()))
				.andExpect(status().is3xxRedirection())
				.andExpect(redirectedUrlTemplate("/request/{id}", request.getId()));

	}

}
