/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.transaction.Transactional;

import nl.tudelft.labracore.api.SessionControllerApi;
import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.queue.TestQueueApplication;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.dto.create.LabCreateDTO;
import nl.tudelft.queue.dto.create.labs.RegularLabCreateDTO;
import nl.tudelft.queue.dto.create.labs.SlottedLabCreateDTO;
import nl.tudelft.queue.dto.patch.LabPatchDTO;
import nl.tudelft.queue.dto.patch.labs.RegularLabPatchDTO;
import nl.tudelft.queue.model.Lab;
import nl.tudelft.queue.model.embeddables.AllowedRequest;
import nl.tudelft.queue.model.embeddables.Slot;
import nl.tudelft.queue.model.embeddables.SlottedLabConfig;
import nl.tudelft.queue.model.enums.CommunicationMethod;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.RegularLab;
import nl.tudelft.queue.model.labs.SlottedLab;
import nl.tudelft.queue.repository.LabRepository;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.modelmapper.ModelMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.ui.Model;

import reactor.core.publisher.Mono;
import test.BaseMockConfig;

@Transactional
@SpringBootTest(classes = TestQueueApplication.class)
@ContextConfiguration(classes = BaseMockConfig.class)
class LabServiceTest {
	@Autowired
	private ModelMapper mapper;

	@Autowired
	private LabService ls;

	@MockBean
	private TimeSlotService tss;

	@Autowired
	private LabRepository lr;

	@Autowired
	private SessionControllerApi sApi;

	@Autowired
	private SessionCacheManager sCache;

	private SessionDetailsDTO session1;
	private SessionDetailsDTO session2;

	private Lab lab;

	private LabCreateDTO createDTO;
	private LabPatchDTO labPatchDTO;

	private Model model;

	@BeforeEach
	void setUp() {
		model = mock(Model.class);

		session1 = new SessionDetailsDTO()
				.rooms(List.of(new RoomSummaryDTO().id(76832L)))
				.edition(new EditionSummaryDTO().id(78932L));

		session2 = new SessionDetailsDTO()
				.editionCollection(new EditionCollectionSummaryDTO().id(6732L));

		lab = RegularLab.builder()
				.communicationMethod(CommunicationMethod.JITSI_MEET)
				.session(6732L)
				.allowedRequests(Set.of(new AllowedRequest(82L, RequestType.QUESTION)))
				.build();

		createDTO = RegularLabCreateDTO.builder()
				.communicationMethod(CommunicationMethod.JITSI_MEET)
				.modules(Set.of(1L))
				.name("Lab 1")
				.requestTypes(Map.of(
						1232L, Set.of(RequestType.QUESTION)))
				.rooms(Set.of(7842L))
				.slot(Slot.builder()
						.opensAt(LocalDateTime.now())
						.closesAt(LocalDateTime.now().plusHours(1L))
						.build())
				.build();

		labPatchDTO = RegularLabPatchDTO.builder()
				.communicationMethod(CommunicationMethod.STUDENT_VISIT_TA)
				.slot(Slot.builder()
						.opensAt(LocalDateTime.now())
						.closesAt(LocalDateTime.now().plusHours(1))
						.build())
				.build();
	}

	@Test
	void setOrganizationInModelSetsEdition() {
		ls.setOrganizationInModel(session1, model);

		verify(model).addAttribute(eq("edition"), any());
	}

	@Test
	void setOrganizationInModelDoesNotSetEditionCollection() {
		ls.setOrganizationInModel(session1, model);

		verify(model).addAttribute("ec", null);
	}

	@Test
	void setOrganizationInModelSession2SetsEditionCollection() {
		ls.setOrganizationInModel(session2, model);

		verify(model).addAttribute(eq("ec"), any());
	}

	@Test
	void setOrganizationInModelSession2DoesNotSetEdition() {
		ls.setOrganizationInModel(session2, model);

		verify(model).addAttribute("edition", null);
	}

	@Test
	void setOrganizationInModelWithLabLooksUpSession() {
		when(sCache.getOrThrow(lab.getSession()))
				.thenReturn(session1);

		ls.setOrganizationInModel(lab, model);

		verify(sCache).getOrThrow(lab.getSession());
	}

	@Test
	void containsRoomFindsExistingRoom() {
		assertThat(ls.containsRoom(session1, 76832L)).isTrue();
	}

	@Test
	void containsRoomDoesNotFindNonExistantRoom() {
		assertThat(ls.containsRoom(session1, 7831237L)).isFalse();
	}

	@Test
	void containsAssignmentFindsExistingAssignment() {
		assertThat(ls.containsAssignment(lab, 82L)).isTrue();
	}

	@Test
	void containsAssignmentDoesNotFindNonExistantAssignment() {
		assertThat(ls.containsAssignment(lab, 89264L)).isFalse();
	}

	@Test
	void containsAllowedRequestFindsExisting() {
		assertThat(ls.containsAllowedRequest(lab, 82L, RequestType.QUESTION)).isTrue();
	}

	@Test
	void containsAllowedRequestNonExistingType() {
		assertThat(ls.containsAllowedRequest(lab, 82L, RequestType.SUBMISSION)).isFalse();
	}

	@Test
	void containsAllowedRequestNonExistingSubmission() {
		assertThat(ls.containsAllowedRequest(lab, 78323L, RequestType.QUESTION)).isFalse();
	}

	@Test
	void createLabSavesANewLab() {
		when(sApi.addSingleSession(any())).thenReturn(Mono.just(5372L));
		when(sApi.addSharedSession(any())).thenReturn(Mono.just(73298L));

		Lab created = ls.createLab(createDTO, 7832L, LabService.LabType.REGULAR);

		assertThat(lr.findById(created.getId())).isPresent().hasValue(created);
	}

	@Test
	void createLabSavesALabUnderRegularSessionId() {
		when(sApi.addSingleSession(any())).thenReturn(Mono.just(5372L));
		when(sApi.addSharedSession(any())).thenReturn(Mono.just(73298L));

		Lab created = ls.createLab(createDTO, 7832L, LabService.LabType.REGULAR);

		assertThat(created.getSession()).isEqualTo(5372L);
	}

	@Test
	void createLabSavesALabUnderSharedSessionId() {
		when(sApi.addSingleSession(any())).thenReturn(Mono.just(5372L));
		when(sApi.addSharedSession(any())).thenReturn(Mono.just(73298L));

		Lab created = ls.createLab(createDTO, 7832L, LabService.LabType.SHARED);

		assertThat(created.getSession()).isEqualTo(73298L);
	}

	@Test
	void createSlottedLabCallsSlotCreation() {
		when(sApi.addSingleSession(any())).thenReturn(Mono.just(5372L));
		when(sApi.addSharedSession(any())).thenReturn(Mono.just(73298L));

		var slottedCreateDTO = mapper.map(createDTO, SlottedLabCreateDTO.class);
		slottedCreateDTO.setSlottedLabConfig(SlottedLabConfig.builder()
				.capacity(3)
				.duration(15L)
				.selectionOpensAt(LocalDateTime.now())
				.build());

		SlottedLab created = ls.createLab(slottedCreateDTO, 7832L, LabService.LabType.REGULAR);

		assertThat(created).isInstanceOf(SlottedLab.class);
		verify(tss).createSlotsForLab(slottedCreateDTO, created);
	}

	@Test
	void updateLabSendsAPatchRequest() {
		when(sApi.patchSession(any(), any())).thenReturn(Mono.empty());

		ls.updateLab(labPatchDTO, lab);

		verify(sApi).patchSession(any(), any());
	}

	@Test
	void updateLabChangesInMemoryLabObject() {
		when(sApi.patchSession(any(), any())).thenReturn(Mono.empty());

		ls.updateLab(labPatchDTO, lab);

		assertThat(lab.getCommunicationMethod()).isEqualTo(CommunicationMethod.STUDENT_VISIT_TA);
	}

	@Test
	void updateLabIncludesAssignmentsIfNonEmpty() {
		when(sApi.patchSession(any(), any())).thenReturn(Mono.empty());

		labPatchDTO.setRequestTypes(Map.of(7832L, Set.of(RequestType.QUESTION)));

		ls.updateLab(labPatchDTO, lab);

		verify(sApi).patchSession(any(), eq(new SessionPatchDTO()
				.name(labPatchDTO.getName())
				.start(labPatchDTO.getSlot().getOpensAt())
				.end(labPatchDTO.getSlot().getClosesAt())
				.assignments(List.of(new AssignmentIdDTO().id(7832L)))
				.rooms(null)));
	}

	@Test
	void updateLabIncludesRoomsIfNonEmpty() {
		when(sApi.patchSession(any(), any())).thenReturn(Mono.empty());

		labPatchDTO.setRooms(Set.of(879342L));

		ls.updateLab(labPatchDTO, lab);

		verify(sApi).patchSession(any(), eq(new SessionPatchDTO()
				.name(labPatchDTO.getName())
				.start(labPatchDTO.getSlot().getOpensAt())
				.end(labPatchDTO.getSlot().getClosesAt())
				.assignments(null)
				.rooms(List.of(new RoomIdDTO().id(879342L)))));
	}

	@Test
	void deleteLab() {
		ls.deleteLab(lab);
		assertThat(lab.getDeletedAt()).isNotNull();
	}
}
