/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.cache;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import nl.tudelft.labracore.api.StudentGroupControllerApi;
import nl.tudelft.labracore.api.dto.RolePersonLayer1DTO;
import nl.tudelft.labracore.api.dto.StudentGroupDetailsDTO;
import nl.tudelft.labracore.api.dto.StudentGroupSummaryDTO;
import nl.tudelft.librador.cache.CoreCacheManager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Component;
import org.springframework.web.context.annotation.RequestScope;

@Component
@RequestScope
@NoArgsConstructor
@AllArgsConstructor
public class StudentGroupCacheManager extends CoreCacheManager<Long, StudentGroupDetailsDTO> {
	private final Map<Long, List<StudentGroupDetailsDTO>> attemptedPersons = new HashMap<>();

	@Autowired
	private StudentGroupControllerApi api;

	@Autowired
	private PersonCacheManager pCache;

	/**
	 * Gets all student groups of a single person by their id.
	 *
	 * @param  personId The id of the person to lookup student groups for.
	 * @return          The list of student group details fetched for this person.
	 */
	public List<StudentGroupDetailsDTO> getByPerson(Long personId) {
		if (attemptedPersons.containsKey(personId)) {
			return attemptedPersons.get(personId);
		}

		var sgs = fetch(api.getGroupsForPerson(personId)
				.map(StudentGroupSummaryDTO::getId)
				.collectList().block());
		attemptedPersons.put(personId, sgs);

		return sgs;
	}

	@Override
	protected List<StudentGroupDetailsDTO> fetch(@Nullable List<Long> ids) {
		if (ids == null || ids.isEmpty()) {
			return List.of();
		}
		return api.getStudentGroupsById(ids).collectList().block();
	}

	@Override
	protected Long getId(StudentGroupDetailsDTO dto) {
		return dto.getId();
	}

	@Override
	protected void registerAdditionally(StudentGroupDetailsDTO dto) {
		pCache.register(Objects.requireNonNull(dto.getMembers()).stream()
				.map(RolePersonLayer1DTO::getPerson).collect(Collectors.toList()));
	}
}
