/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.dto.view;

import static java.time.LocalDateTime.now;
import static nl.tudelft.librador.SpringContext.getBean;

import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import nl.tudelft.labracore.api.dto.EditionSummaryDTO;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.cache.EditionCollectionCacheManager;
import nl.tudelft.queue.cache.SessionCacheManager;
import nl.tudelft.queue.misc.QueueSessionStatus;
import nl.tudelft.queue.model.QueueSession;
import nl.tudelft.queue.model.TimeSlot;
import nl.tudelft.queue.model.embeddables.Slot;
import nl.tudelft.queue.model.enums.QueueSessionType;
import nl.tudelft.queue.model.labs.Lab;
import nl.tudelft.queue.model.labs.SlottedLab;
import nl.tudelft.queue.service.LabService;

@Data
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = false)
public class QueueSessionSummaryDTO extends View<QueueSession<?>> {
	private Long id;

	private QueueSessionType type;

	private List<EditionSummaryDTO> associatedEditions;

	private String name;
	private String readableName;
	private Slot slot;

	private Integer slotsOccupied;
	private String slotOccupationString = "";

	private Boolean isSlotSelectionOpen = false;
	private Boolean isActiveOrGracePeriod = false;

	private QueueSessionStatus status;

	private Boolean isShared = false;

	@Override
	public void postApply() {
		var session = getBean(SessionCacheManager.class).getRequired(data.getSession());

		this.name = session.getName();
		this.readableName = session.getName() + " : " + DateTimeFormatter.ofPattern("HH:mm - dd " +
				"MMM yyyy").format(session.getStart());
		this.slot = new Slot(session.getStart(), session.getEndTime());

		this.status = getBean(LabService.class).getLabStatus(data);

		isActiveOrGracePeriod = slot.open();
		if (data instanceof Lab) {
			isActiveOrGracePeriod = slot.openWithGracePeriod(((Lab) data).getEolGracePeriod());
		}

		if (data instanceof SlottedLab) {
			slotsOccupied = slotsOccupied((SlottedLab) data);
			slotOccupationString = slotOccupationString((SlottedLab) data);

			isSlotSelectionOpen = ((SlottedLab) data).getSlottedLabConfig().getSelectionOpensAt()
					.isBefore(now());
		}
		isShared = session.getEditionCollection() != null;
		if (isShared) {
			var editionCollection = getBean(EditionCollectionCacheManager.class)
					.getRequired(session.getEditionCollection().getId());
			associatedEditions = editionCollection.getEditions();
		} else {
			associatedEditions = List.of(Objects.requireNonNull(session.getEdition()));
		}

	}

	/**
	 * Gets the string describing the current slot occupation of the given lab.
	 *
	 * @param  lab The lab to calculate slot occupation over.
	 * @return     The string representing slot occupation (empty string for non-slotted labs).
	 */
	private String slotOccupationString(SlottedLab lab) {
		long slotsTotal = lab.getTimeSlots().stream().mapToInt(TimeSlot::getCapacity).sum();
		long slotsTaken = lab.getTimeSlots().stream().mapToInt(TimeSlot::countSlotsOccupied).sum();

		return "(" + slotsTaken + "/" + slotsTotal + " slots taken)";
	}

	/**
	 * Gets the current number of slots occupied by student requests in this lab.
	 *
	 * @param  lab The lab to find the number of occupied slots for.
	 * @return     The number of slots in this lab that are currently occupied.
	 */
	private Integer slotsOccupied(SlottedLab lab) {
		return lab.getTimeSlots().stream().mapToInt(TimeSlot::countSlotsOccupied).sum();
	}

	public QueueSession<?> getData() {
		return this.data;
	}

}
