/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model.constraints;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.OneToOne;
import javax.validation.constraints.NotNull;

import lombok.*;
import lombok.experimental.SuperBuilder;
import nl.tudelft.labracore.api.ModuleDivisionControllerApi;
import nl.tudelft.labracore.api.dto.StudentGroupDetailsDTO;
import nl.tudelft.librador.SpringContext;
import nl.tudelft.queue.cache.PersonCacheManager;
import nl.tudelft.queue.cache.StudentGroupCacheManager;
import nl.tudelft.queue.dto.create.constraints.ModuleDivisionConstraintCreateDTO;
import nl.tudelft.queue.model.LabRequestConstraint;
import nl.tudelft.queue.model.QueueSession;
import nl.tudelft.queue.model.Request;

@Data
@Entity
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class ModuleDivisionConstraint extends LabRequestConstraint {
	/**
	 * The set of division ids from which groups are allowed to make requests.
	 */
	@Builder.Default
	@ElementCollection
	private Set<Long> divisions = new HashSet<>();

	/**
	 * The Lab that this constraint is linked to.
	 */
	@NotNull
	@OneToOne
	private QueueSession<?> session;

	@Override
	public boolean canCreateRequest(Long personId) {
		// Go through every one of the student groups the person is in and check whether they are in one that is part
		// of an allowed module division.
		Stream<Long> studentGroups = SpringContext.getBean(StudentGroupCacheManager.class)
				.getByPerson(personId).stream().map(StudentGroupDetailsDTO::getId);
		return studentInDivision(personId)
				|| studentGroupInDivision(studentGroups);
	}

	@Override
	public boolean allowsRequest(Request<?> request) {
		// Retrieve the student group for the request.

		return studentInDivision(request.getRequester())
				|| studentGroupInDivision(Stream.of(request.getStudentGroup()));
	}

	/**
	 * Check if student is in the division.
	 *
	 * @param  personId Person for whom to check.
	 * @return          {@code True} if the person is in the division.
	 */
	private boolean studentInDivision(Long personId) {
		var person = SpringContext.getBean(PersonCacheManager.class).get(personId).orElseThrow();
		var moduleApi = SpringContext.getBean(ModuleDivisionControllerApi.class);
		return divisions.stream().anyMatch(d -> Objects
				.requireNonNull(moduleApi.getPeopleInDivision(d).block()).getPeople().contains(person));
	}

	/**
	 * Check if a student group is in the division.
	 *
	 * @param  studentGroup The student group for whom to check.
	 * @return              {@code True} if the student group is in the division.
	 */
	private boolean studentGroupInDivision(Stream<Long> studentGroup) {
		var sgo = SpringContext
				.getBean(StudentGroupCacheManager.class);
		//				.get(studentGroup);

		return studentGroup.map(s -> sgo.get(s).orElseThrow())
				.anyMatch(sg -> sg.getDivision() != null && divisions.contains(sg.getDivision().getId()));
	}

	@Override
	public String displayName() {
		return "Module Division";
	}

	@Override
	public String constraintDescription() {
		return "in division " + divisions.stream().map(div -> "#" + div).collect(Collectors.joining(", "));
	}

	@Override
	public ModuleDivisionConstraintCreateDTO toCreateDTO() {
		return new ModuleDivisionConstraintCreateDTO(new HashSet<>(divisions));
	}
}
