/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model.labs;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.OptionalDouble;

import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;
import nl.tudelft.labracore.api.dto.SessionDetailsDTO;
import nl.tudelft.librador.dto.view.View;
import nl.tudelft.queue.dto.create.labs.LabCreateDTO;
import nl.tudelft.queue.dto.create.labs.RegularLabCreateDTO;
import nl.tudelft.queue.dto.patch.LabPatchDTO;
import nl.tudelft.queue.dto.patch.labs.RegularLabPatchDTO;
import nl.tudelft.queue.dto.view.labs.RegularLabViewDTO;
import nl.tudelft.queue.model.LabRequest;
import nl.tudelft.queue.model.enums.QueueSessionType;

@Data
@Entity
@SuperBuilder
@NoArgsConstructor
@DiscriminatorValue("REGULAR")
@EqualsAndHashCode(callSuper = true)
public class RegularLab extends Lab {

	@Override
	protected QueueSessionType getLabType() {
		return QueueSessionType.REGULAR;
	}

	@Override
	public LabCreateDTO<?> copyLabCreateDTO(SessionDetailsDTO session) {
		return new RegularLabCreateDTO(session, this);
	}

	@Override
	public LabPatchDTO<?> newPatchDTO() {
		return new RegularLabPatchDTO();
	}

	@Override
	public RegularLabViewDTO toViewDTO() {
		return View.convert(this, RegularLabViewDTO.class);
	}

	@Override
	public OptionalDouble currentWaitingTime() {
		var labEndTime = getSessionDto().getEndTime();

		if (labEndTime.isBefore(LocalDateTime.now())) {
			// Lab is over, display waiting time of current requests or else 0
			return getPendingRequests().stream()
					.min(Comparator.comparing(LabRequest::getCreatedAt))
					.map(r -> r.sensibleWaitingTime(labEndTime))
					.orElse(OptionalDouble.of(0));
		} else {
			// Lab is running. Get the waiting time of the oldest pending request.
			// If nothing is pending, report the waiting time of the last handled request.
			// If nothing was handled, report no waiting time.
			return getPendingRequests().stream()
					.min(Comparator.comparing(LabRequest::getCreatedAt))
					.map(LabRequest::sensibleWaitingTime)
					.orElseGet(() -> getHandled().stream()
							.max(Comparator.comparing(LabRequest::getCreatedAt))
							.map(LabRequest::sensibleWaitingTime)
							.orElse(OptionalDouble.empty()));
		}

	}
}
