/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.repository;

import java.time.LocalDateTime;
import java.util.List;

import nl.tudelft.queue.model.misc.Announcement;
import nl.tudelft.queue.model.misc.QAnnouncement;

import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.querydsl.QuerydslPredicateExecutor;
import org.springframework.lang.NonNull;

import com.querydsl.core.types.Predicate;

public interface AnnouncementRepository extends
		JpaRepository<Announcement, Long>,
		QuerydslPredicateExecutor<Announcement>,
		ThrowingRepository<Announcement, Long> {
	QAnnouncement qa = QAnnouncement.announcement;

	@NonNull
	@Override
	List<Announcement> findAll(@NonNull Predicate predicate);

	/**
	 * Finds all announcements that should currently be displayed.
	 *
	 * @return The list of active announcements.
	 */
	default List<Announcement> findActiveAnnouncements() {
		return findAll(qa.startTime.before(LocalDateTime.now())
				.andAnyOf(qa.endTime.isNull(), qa.endTime.after(LocalDateTime.now())));
	}

	/**
	 * Finds all announcements that can be edited by an admin of the Queue. Any announcement that is not
	 * editable can be removed or forgotten. We can still keep them in the database for looking back on when
	 * maintenance was done etc.
	 *
	 * @return The list of announcements that can still be edited by an admin.
	 */
	default List<Announcement> findEditableAnnouncements() {
		return findAll(qa.endTime.isNull().or(qa.endTime.after(LocalDateTime.now())));
	}
}
