/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import nl.tudelft.queue.model.enums.RequestStatus;
import nl.tudelft.queue.model.enums.SelectionProcedure;
import nl.tudelft.queue.model.labs.CapacitySession;
import nl.tudelft.queue.model.labs.QCapacitySession;

public interface CapacitySessionRepository extends QueueRepository<CapacitySession, Long> {
	QCapacitySession qcs = QCapacitySession.capacitySession;

	/**
	 * @return All capacity labs that should select the students that may go to the session.
	 */
	default List<CapacitySession> findAllThatShouldPickStudents() {
		LocalDateTime now = LocalDateTime.now();
		return findAll(qcs.capacitySessionConfig.selectionAt.before(now)
				.and(qcs.requests.any().eventInfo.status.eq(RequestStatus.PENDING))
				.and(qcs.capacitySessionConfig.procedure.ne(SelectionProcedure.FCFS)));
	}

	/**
	 * @param  moduleIds The module ids to find labs for.
	 * @return           All capacity labs that covered any of the given list of modules.
	 */
	default List<CapacitySession> findAllPastLabsByModules(Collection<Long> moduleIds) {
		return findAll(qcs.capacitySessionConfig.selectionAt.before(LocalDateTime.now())
				.and(qcs.modules.any().in(moduleIds))
				.and(qcs.capacitySessionConfig.procedure.ne(SelectionProcedure.FCFS)));
	}
}
