/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2021  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.queue.dto.create.CourseRequestCreateDTO;
import nl.tudelft.queue.dto.util.ReportDTO;
import nl.tudelft.queue.properties.MailProperties;
import nl.tudelft.queue.properties.QueueProperties;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.mail.MailSender;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.stereotype.Service;

@Service
@ConditionalOnProperty(name = "queue.mail.enabled", havingValue = "true")
public class MailService {
	@Autowired
	private QueueProperties qp;

	@Autowired
	private MailProperties mp;

	@Autowired
	private MailSender mailSender;

	/**
	 * Sends an e-mail to the Queue administration e-mail to request a new course.
	 *
	 * @param person The person that is requesting the new course.
	 * @param dto    The DTO containing the request for the new course.
	 */
	public void sendCourseRequest(Person person, CourseRequestCreateDTO dto) {
		var msg = new SimpleMailMessage();
		msg.setTo(mp.getSupportEmail());
		msg.setFrom(mp.getOriginEmail());
		msg.setReplyTo(person.getEmail());

		msg.setSubject("Request to create course: " + dto.getName() + " (" + dto.getCode() + ")");
		msg.setText("user-displayname: " + person.getDisplayName() +
				"\nuser-username: " + person.getUsername() +
				"\nuser-id: " + person.getId() +
				"\nuser-email: " + person.getEmail() +
				"\nname: " + dto.getName() +
				"\ncode: " + dto.getCode() +
				"\nextra-info: " + dto.getRemarks() +
				"\n\nlink: " + qp.getDeployUrl().replaceAll("^(.+)/$", "\1") +
				"/admin/course/add?name=" + dto.getName().replaceAll(" ", "+") +
				"&code=" + dto.getCode().replaceAll(" ", "+") +
				"&manager=" + person.getUsername().replaceAll(" ", "+"));

		mailSender.send(msg);
	}

	/**
	 * Send an email about the report
	 *
	 * @param person The person who is filling the report.
	 * @param report The report for which the email should be send.
	 */
	public void sendReportMail(Person person, ReportDTO report) {
		var msg = new SimpleMailMessage();
		msg.setTo(report.getRecipients());
		msg.setFrom(mp.getOriginEmail());
		StringBuilder content = new StringBuilder();
		content.append(String.format("A new report has been submitted about %s ",
				report.getAssistantName()));
		if (report.isAnonymous()) {
			msg.setReplyTo(mp.getOriginEmail());
			msg.setSubject("An anonymous report has been submitted through Queue");
		} else {
			msg.setReplyTo(person.getEmail());
			msg.setSubject("A report has been submitted through Queue");
			content.append(String.format("by %s ", person.getDisplayName()));
		}
		content.append(String.format("\n\n The following reason was given: %s",
				report.getReason()));
		msg.setText(content.toString());
		mailSender.send(msg);
	}
}
