/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

let colorIdx = 0;

/**
 * Color-blind friendly random colors used in graphs.
 *
 * @param idx {number} If provided, will correspond to the idx color in the dictionary
 * @returns {string} A random color that is guaranteed not to be the same on successive calls.
 */
function getArbitraryColorForGraph(idx = null) {
    const colorPalette = {
        BLUE: "rgba(0,114,178,0.7)",
        VERMILION: "rgba(213,94,0,0.7)",
        REDDISH_PINK: "rgba(204,121,167,0.7)",
        TEAL: "rgba(0,158,115,0.7)",
        ORANGE: "rgba(230,159,0,0.7)",
    };

    const colors = Object.values(colorPalette);

    if (idx !== null) return colors[idx % colors.length];

    return colors[colorIdx++ % colors.length];
}

function getBorderColor() {
    return "rgba(0,0,0,0.7)";
}

/**
 * Dictionary for status colors to be used for graphs.
 *
 * @param status The status you want to retrieve the color for
 * @returns {string} The associated color
 */
function getColorForStatus(status) {
    const colorsForStatus = {
        PENDING: "rgba(0, 123, 255, 0.7)",
        PROCESSING: "rgba(23, 162, 184, 0.7)",
        PROCESSED: "rgba(141,102,238,0.7)",
        TOTAL: "rgba(230,159,0,0.7)",
        APPROVED: "rgba(40, 167, 69, 0.7)",
        REJECTED: "rgba(255, 99, 132, 0.7)",
        NOT_FOUND: "rgba(255, 206, 86, 0.7)",
        REVOKED: "rgba(108, 117, 125, 0.7)",
        QUESTION: "rgba(0, 123, 255, 0.7)",
        SUBMISSION: "rgba(230,159,0,0.7)",
    };

    return colorsForStatus[status];
}

/**
 * Function which takes ms and converts it to a nice time display
 *
 * @param ms The amount of ms that the duration covers
 * @returns {string} A string which has a human representation of the spanned time.
 */
function msToHumanReadableTime(ms) {
    let seconds = (ms / 1000).toFixed(1);
    let minutes = (ms / (1000 * 60)).toFixed(1);
    let hours = (ms / (1000 * 60 * 60)).toFixed(1);
    let days = (ms / (1000 * 60 * 60 * 24)).toFixed(1);
    if (seconds < 60) return seconds + " Seconds";
    else if (minutes < 60) return minutes + " Minutes";
    else if (hours < 24) return hours + " Hours";
    else return days + " Days";
}
