/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.dto.create.util;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;

import java.time.LocalDateTime;
import java.util.Set;
import java.util.stream.Stream;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import nl.tudelft.queue.dto.create.labs.CapacitySessionCreateDTO;
import nl.tudelft.queue.dto.create.labs.RegularLabCreateDTO;
import nl.tudelft.queue.dto.create.labs.SlottedLabCreateDTO;
import nl.tudelft.queue.dto.util.ImportSessionDTO;
import nl.tudelft.queue.model.embeddables.Slot;
import nl.tudelft.queue.model.enums.QueueSessionType;
import nl.tudelft.queue.model.labs.CapacitySession;
import nl.tudelft.queue.model.labs.RegularLab;
import nl.tudelft.queue.model.labs.SlottedLab;

public class ImportSessionDTOTest {

	ImportSessionDTO test;

	@BeforeEach
	void setup() {
		test = new ImportSessionDTO("NAME", LocalDateTime.now(),
				LocalDateTime.now().plusHours(4L), QueueSessionType.REGULAR, Set.of(4L));
	}

	@Test
	void testGetCreateDTORegular() {
		var dto = new RegularLabCreateDTO();
		dto.setName("NAME");
		dto.setRooms(Set.of(4L));
		dto.setSlot(new Slot(test.getStart(), test.getEnd()));
		assertThat(test.getCreateDTO()).isEqualTo(dto);
	}

	@Test
	void testGetCreateDTOSlotted() {
		test.setSessionType(QueueSessionType.SLOTTED);
		var dto = new SlottedLabCreateDTO();
		dto.setName("NAME");
		dto.setRooms(Set.of(4L));
		dto.setSlot(new Slot(test.getStart(), test.getEnd()));
		assertThat(test.getCreateDTO()).isEqualTo(dto);
	}

	@Test
	void testGetCreateDTOCapacity() {
		test.setSessionType(QueueSessionType.CAPACITY);
		var dto = new CapacitySessionCreateDTO();
		dto.setName("NAME");
		dto.setRooms(Set.of(4L));
		dto.setSlot(new Slot(test.getStart(), test.getEnd()));
		assertThat(test.getCreateDTO()).isEqualTo(dto);
	}

	@ParameterizedTest
	@MethodSource("provideParameter")
	void testGetQueueSession(QueueSessionType type, Class clazz) {
		test.setSessionType(type);
		assertThat(test.getQueueSession()).isExactlyInstanceOf(clazz);
	}

	private static Stream<Arguments> provideParameter() {
		return Stream.of(
				Arguments.of(QueueSessionType.REGULAR, RegularLab.class),
				Arguments.of(QueueSessionType.SLOTTED, SlottedLab.class),
				Arguments.of(QueueSessionType.CAPACITY, CapacitySession.class));
	}
}
