/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.controller;

import java.util.stream.Collectors;

import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;

import lombok.RequiredArgsConstructor;
import nl.tudelft.labracore.api.CourseControllerApi;
import nl.tudelft.labracore.api.dto.CourseSummaryDTO;
import nl.tudelft.labracore.api.dto.CourseWithManagerCreateDTO;
import nl.tudelft.labracore.api.dto.PersonIdDTO;
import nl.tudelft.labracore.lib.security.user.AuthenticatedPerson;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.queue.service.CourseService;
import nl.tudelft.queue.service.TuDelftApiService;

@Controller
@RequestMapping("course")
@RequiredArgsConstructor
public class CourseController {

	private final TuDelftApiService tuDelftApiService;

	private final CourseService courseService;

	private final CourseControllerApi courseControllerApi;

	/**
	 * Endpoint to show the teacher his/her courses that don't have a corresponding labrador course.
	 *
	 * @param  person The teacher
	 * @param  model  The model
	 * @return        View responsible for displaying the qualifying courses.
	 */
	@GetMapping("my-courses")
	@PreAuthorize("@permissionService.isAdminOrTeacher()")
	public String showMyCourses(@AuthenticatedPerson Person person, Model model) {
		var managedCourses = courseControllerApi.getAllCoursesByManager(person.getId()).collectList().block()
				.stream().map(CourseSummaryDTO::getCode).collect(Collectors.toSet());
		model.addAttribute("courses",
				tuDelftApiService.getCoursesForInstructor(person)
						.stream().filter((dto) -> !managedCourses.contains(dto.getCode())).distinct()
						.toList());
		model.addAttribute("create", new CourseWithManagerCreateDTO());
		return "course/my-courses";
	}

	/**
	 * POST endpoint to allow a teacher to create a labracore course based off information from studiegids.
	 *
	 * @param  person          The teacher
	 * @param  courseCreateDTO The DTO corresponding to the submitted form
	 * @return                 The view where the teacher can add editions of new courses.
	 */
	@PostMapping("create")
	@PreAuthorize("@permissionService.isAdminOrTeacher()")
	public String createCourse(@AuthenticatedPerson Person person,
			CourseWithManagerCreateDTO courseCreateDTO) {
		courseService.setupCourseWithManagerLabracore(
				courseCreateDTO.manager(new PersonIdDTO().id(person.getId())));
		return "redirect:/edition/add";
	}
}
