/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.model;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.OptionalDouble;

import jakarta.persistence.*;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Size;

import com.fasterxml.jackson.annotation.JsonIgnore;

import lombok.*;
import lombok.experimental.SuperBuilder;
import nl.tudelft.queue.dto.view.requests.LabRequestViewDTO;
import nl.tudelft.queue.model.enums.Language;
import nl.tudelft.queue.model.enums.OnlineMode;
import nl.tudelft.queue.model.enums.RequestType;
import nl.tudelft.queue.model.labs.ExamLab;
import nl.tudelft.queue.model.labs.Lab;

/**
 * A request that has been made by a StudentGroup, that the database representation of someone asking for
 * help. Part of this class stores it's data directly in the Request table. The lifecycle of a Request, that
 * is being: created, reassigned, forwarded, approved, etc; is done through event sourcing
 * {@link RequestEvent}. Every change in the requests lifecycle append a new event to this log.
 * <p>
 * The entire log is then reran when the Request is recreated by hibernate through an {@link PostLoad} hook,
 * which fills in the remaining transient (not persisted) fields of this class.
 */
@Data
@Entity
@SuperBuilder
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
public class LabRequest extends Request<Lab> {
	/**
	 * The type of the request (either a question or a submission).
	 */
	@NotNull
	private RequestType requestType;

	/**
	 * The question a student placed to describe the intent of their request.
	 */
	@Lob
	@Size(max = 500)
	private String question;

	/**
	 * The Jitsi room this request will take place in.
	 */
	private String jitsiRoom;

	/**
	 * The OnlineMode this request will take place in (Jitsi, MS teams, etc)
	 */
	private OnlineMode onlineMode;

	/**
	 * The language of the request.
	 */
	@NotNull
	@Builder.Default
	@Enumerated(EnumType.STRING)
	private Language language = Language.ANY;

	/**
	 * The timeslot this request occupies. Timeslots can be used as opposed to direct requests to allow for
	 * reserving a TA for a specific time ahead of the lab starting.
	 */
	@ManyToOne
	@JsonIgnore
	private TimeSlot timeSlot;

	/**
	 * The Assignment that this request is asking a question about or checking a submission of.
	 */
	@NotNull
	private Long assignment;

	private Long questionId;

	/**
	 * The feedback given through this Request.
	 */
	@JsonIgnore
	@Builder.Default
	@ToString.Exclude
	@EqualsAndHashCode.Exclude
	@OneToMany(mappedBy = "request")
	private List<Feedback> feedbacks = new ArrayList<>();

	/**
	 * Gets the waiting time in seconds that is being experienced, or was experienced for this request. The
	 * waiting time is defined differently depending on the specifics of the request:
	 * <ul>
	 * <li>For once-processed, non-timeslotted requests: the time between the creation of the request and the
	 * first time that the request was processed.</li>
	 * <li>For once-processed, timeslotted requests: the time between the start of the timeslot and the first
	 * time that the request was processed.</li>
	 * <li>For unprocessed, non-timeslotted requests: the time between the creation of the request and
	 * effectiveEndTime.</li>
	 * <li>For unprocessed, timeslotted requests: the time between the start of the timeslot and
	 * effectiveEndTime.</li>
	 * </ul>
	 *
	 * If this request has a timeslot which starts in the future, this method returns an empty OptionalLong.
	 * We also consider the end time of the session to override the effectiveEndTime so that viewing wait time
	 * statistics for a session that occurs in the past is not skewed.
	 *
	 * @param  effectiveEndTime The effective end time to collect wait time statistics. Usually this is the
	 *                          present or the end of the lab. if computing bucketed statistics, you can put
	 *                          it a time in the past.
	 * @param  unit             The unit you want to compute this time difference in.
	 * @return                  The waiting time in the desired unit.
	 */
	public OptionalDouble waitTime(LocalDateTime effectiveEndTime,
			ChronoUnit unit) {
		LocalDateTime startTime;
		if (getTimeSlot() == null) {
			startTime = getCreatedAt();
		} else {
			startTime = getTimeSlot().getSlot().getOpensAt();
		}

		LocalDateTime endTime;
		if (getEventInfo().getFirstProcessedAt() == null) {
			endTime = effectiveEndTime;
		} else {
			endTime = getEventInfo().getFirstProcessedAt();
		}

		return calculateTimeDifferenceWithUnit(startTime, endTime, unit);
	}

	/**
	 * The processing time of a request. If a request has not yet processed, it is not considered and will
	 * return an empty OptionalLong This also occurs when looking at something in the future. It contributes
	 * to the calculation if it is currently being processed, or if it has been processed.
	 *
	 *
	 * @param  effectiveEndTime The effective end time to collect wait time statistics. Usually this is the
	 *                          present or the end of the lab. if computing bucketed statistics, you can put
	 *                          it a time in the past.
	 * @param  unit             The unit you want to compute this time difference in.
	 * @return                  The processing time of the current request.
	 */
	public OptionalDouble processingTime(LocalDateTime effectiveEndTime,
			ChronoUnit unit) {

		final LocalDateTime startTime = getEventInfo().getFirstProcessedAt();

		if (startTime == null)
			return OptionalDouble.empty();

		LocalDateTime endTime;
		if (getEventInfo().getHandledAt() == null) {
			endTime = effectiveEndTime;
		} else {
			endTime = getEventInfo().getHandledAt();
		}

		return calculateTimeDifferenceWithUnit(startTime, endTime, unit);
	}

	@Override
	public boolean isRevokable() {
		boolean isRevokable = true;
		if (getSession() instanceof ExamLab) {
			isRevokable = !getSession().getEnqueueClosed();
		}
		return isRevokable && getEventInfo().getStatus().isPending();
	}

	@Override
	public Class<LabRequestViewDTO> viewClass() {
		return LabRequestViewDTO.class;
	}

	/**
	 * Calculate a time difference with a specified unit.
	 *
	 * @param  start The lower bound
	 * @param  end   The upper bound
	 * @param  unit  The unit you wish to retrieve the time difference in
	 * @return       The time difference in the specified unit, if end < start, it will return an empty
	 *               optional
	 */
	private OptionalDouble calculateTimeDifferenceWithUnit(LocalDateTime start, LocalDateTime end,
			ChronoUnit unit) {
		long time = unit.between(start, end);
		if (time < 0) {
			return OptionalDouble.empty();
		}

		return OptionalDouble.of(time);
	}
}
