/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import nl.tudelft.labracore.api.dto.AssignmentDetailsDTO;
import nl.tudelft.labracore.api.dto.AssignmentSummaryDTO;
import nl.tudelft.labracore.api.dto.CourseSummaryDTO;
import nl.tudelft.labracore.api.dto.EditionDetailsDTO;
import nl.tudelft.labracore.api.dto.EditionSummaryDTO;
import nl.tudelft.labracore.api.dto.ModuleDetailsDTO;
import nl.tudelft.queue.cache.AssignmentCacheManager;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.ModuleCacheManager;

@Service
public class AssignmentService {

	@Autowired
	private ModuleCacheManager mCache;

	@Autowired
	private AssignmentCacheManager aCache;

	@Autowired
	private EditionCacheManager eCache;

	@Autowired
	private RoleDTOService roleDTOService;

	@Autowired
	private ModuleCacheManager moduleCache;

	@Autowired
	private EditionCacheManager editionCache;

	/**
	 * Retrieves the course corresponding to a certain assignment. Used in filtering lab requests by the
	 * course they belong to.
	 *
	 * @param  assignment the assignment to retrieve the course for
	 * @return            the course corresponding to the assignment
	 */
	public CourseSummaryDTO getCourseByAssignment(AssignmentDetailsDTO assignment) {
		ModuleDetailsDTO module = moduleCache.getRequired(assignment.getModule().getId());

		EditionDetailsDTO edition = editionCache.getRequired(module.getEdition().getId());

		return edition.getCourse();
	}

	/**
	 * Gets the associated edition of a given assignment.
	 *
	 * @param  assignmentId The assignment in question
	 * @return              The edition that this assignment corresponds to.
	 */
	public EditionSummaryDTO getEditionForAssignment(Long assignmentId) {
		return mCache.getRequired(aCache.getRequired(assignmentId).getModule().getId())
				.getEdition();
	}

	/**
	 * Finds for which of the allowed assignments the assistant is a head TA or teacher
	 *
	 * @param  allowedAssignments the assignments to filter
	 * @param  assistantId        the assistant whose role is of interest
	 * @return                    a filtered list with assignment ids for which the assistant is a teacher or
	 *                            head TA of the edition
	 */
	public List<Long> assignmentsForWhichIsTeacherHeadTA(List<AssignmentSummaryDTO> allowedAssignments,
			Long assistantId) {
		return allowedAssignments
				.stream()
				.map(AssignmentSummaryDTO::getId)
				.filter(assignmentId -> {
					Long eId = getEditionForAssignment(assignmentId).getId();
					return roleDTOService.isTeacherOrHeadTAInEdition(assistantId, eCache.getRequired(eId));
				}).toList();

	}
}
