/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
document.addEventListener("DOMContentLoaded", () => {
    // On clicking interactable maps, update the indicator
    document.querySelectorAll("[data-map][data-interactable]").forEach(imageHolder => {
        imageHolder.addEventListener("click", event => {
            const mapId = imageHolder.getAttribute("data-map");
            if (event.target.tagName === "IMG") {
                const img = event.target;
                const indicatorHolder = img.closest(".indicator-holder");

                const imgBounds = img.getBoundingClientRect();

                const clickX_on_img = event.offsetX;
                const clickY_on_img = event.offsetY;
                const imgOffsetLeft = img.offsetLeft;
                const imgOffsetTop = img.offsetTop;

                const totalClickX_px = imgOffsetLeft + clickX_on_img;
                const totalClickY_px = imgOffsetTop + clickY_on_img;

                const holderWidth = indicatorHolder.clientWidth;
                const holderHeight = indicatorHolder.clientHeight;

                const cssLocation = {
                    x: totalClickX_px / holderWidth,
                    y: totalClickY_px / holderHeight,
                };

                updateIndicator(mapId, cssLocation);
            }
        });
    });

    // When a select with a 'data-update-map' changes, reset the indicator of corresponding maps
    document.querySelectorAll("[data-update-map]").forEach(select => {
        select.addEventListener("change", () => {
            const mapId = select.getAttribute("data-update-map");
            resetIndicator(mapId);
        });
    });
});

function updateIndicator(mapId, cssLocation) {
    // Update all indicators
    document.querySelectorAll(`[data-map="${mapId}"]`).forEach(imageHolder => {
        let indicator = imageHolder.querySelector(".indicator");
        if (!indicator) {
            const indicatorHolder = imageHolder.querySelector(".indicator-holder");
            indicator = document.createElement("div");
            indicator.classList.add("indicator");
            indicatorHolder.appendChild(indicator);
        }

        // Use the CSS-relative location here
        indicator.style.left = `${cssLocation.x * 100}%`;
        indicator.style.top = `${cssLocation.y * 100}%`;
    });

    // Update all x coordinate fields
    document.querySelectorAll(`[data-map-x="${mapId}"]`).forEach(locationX => {
        locationX.name = "location.x";
        locationX.value = cssLocation.x;
    });
    // Update all y coordinate fields
    document.querySelectorAll(`[data-map-y="${mapId}"]`).forEach(locationY => {
        locationY.name = "location.y";
        locationY.value = cssLocation.y;
    });
}

function resetIndicator(mapId) {
    // Reset all indicators
    document.querySelectorAll(`[data-map="${mapId}"]`).forEach(imageHolder => {
        const indicator = imageHolder.querySelector(".indicator");
        if (indicator) {
            indicator.remove();
        }
    });

    // Reset all x coordinate fields
    document.querySelectorAll(`[data-map-x="${mapId}"]`).forEach(locationX => {
        locationX.name = "";
    });
    // Reset all y coordinate fields
    document.querySelectorAll(`[data-map-y="${mapId}"]`).forEach(locationY => {
        locationY.name = "";
    });
}
