/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2024  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.queue.service;

import static org.mockito.Mockito.when;

import java.util.List;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;

import nl.tudelft.labracore.api.dto.*;
import nl.tudelft.labracore.lib.security.user.Person;
import nl.tudelft.queue.cache.AssignmentCacheManager;
import nl.tudelft.queue.cache.EditionCacheManager;
import nl.tudelft.queue.cache.ModuleCacheManager;
import nl.tudelft.queue.model.LabRequest;
import nl.tudelft.queue.model.embeddables.RequestEventInfo;
import nl.tudelft.queue.model.enums.ForwardingGroup;
import nl.tudelft.queue.model.enums.RequestStatus;
import nl.tudelft.queue.model.labs.Lab;
import nl.tudelft.queue.model.labs.RegularLab;
import test.test.TestQueueApplication;

@SpringBootTest(classes = TestQueueApplication.class)
public class ForwardingTest {
	@Autowired
	private RequestService rs;

	@Autowired
	private AssignmentService as;

	@MockBean
	private RoleDTOService roleDTOService;

	@MockBean
	private EditionCacheManager eCache;

	@MockBean
	private AssignmentCacheManager aCache;

	@MockBean
	private ModuleCacheManager mCache;

	private Long teacherId;
	private Long taId;
	private Person teacher;
	private Person ta;

	private LabRequest request;
	private RequestEventInfo eventInfo;
	private AssignmentDetailsDTO assignment;

	@BeforeEach
	void setUp() {
		Long editionId = 1000L;
		Long assignmentId = 2000L;
		Long moduleId = 3000L;
		Long sessionId = 4000L;
		Long requestId = 5000L;
		teacherId = 6000L;
		taId = 7000L;

		teacher = new Person();
		ta = new Person();
		teacher.setId(teacherId);
		ta.setId(taId);

		EditionSummaryDTO edition = new EditionSummaryDTO().id(editionId);
		EditionDetailsDTO editionDetails = new EditionDetailsDTO().id(editionId);

		Lab session = new RegularLab();
		session.setId(sessionId);

		ModuleSummaryDTO module = new ModuleSummaryDTO().id(moduleId);
		ModuleDetailsDTO moduleDetails = new ModuleDetailsDTO().id(moduleId).edition(edition);

		assignment = new AssignmentDetailsDTO().id(assignmentId).module(module);

		eventInfo = new RequestEventInfo();

		request = new LabRequest();
		request.setSession(session);
		request.setId(requestId);
		request.setAssignment(assignmentId);
		request.setEventInfo(eventInfo);

		when(aCache.getRequired(assignmentId)).thenReturn(assignment);
		when(mCache.getRequired(moduleId)).thenReturn(moduleDetails);
		when(eCache.getRequired(editionId)).thenReturn(editionDetails);

		when(roleDTOService.isTeacherOrHeadTAInEdition(teacherId, editionDetails)).thenReturn(Boolean.TRUE);
		when(roleDTOService.isTeacherOrHeadTAInEdition(taId, editionDetails)).thenReturn(Boolean.FALSE);
	}

	@Test
	void testOnlyHeadTAOrTeacherSeeRequestForwardedToThem() {
		Assertions.assertTrue(rs.shouldSeeForwardedToTeacherHeadTA(teacher, request));
		Assertions.assertFalse(rs.shouldSeeForwardedToTeacherHeadTA(ta, request));
	}

	@Test
	void testPendingIsShownToEveryone() {
		eventInfo.setStatus(RequestStatus.PENDING);

		Assertions.assertTrue(rs.shouldShowRequest(request, teacher));
		Assertions.assertTrue(rs.shouldShowRequest(request, ta));
	}

	@Test
	void testForwardedToEveryoneIsShownToEveryone() {
		eventInfo.setStatus(RequestStatus.FORWARDED);
		eventInfo.setForwardingGroup(ForwardingGroup.ANYONE);

		Assertions.assertTrue(rs.shouldShowRequest(request, teacher));
		Assertions.assertTrue(rs.shouldShowRequest(request, ta));
	}

	@Test
	void testForwardedToTeachersOrHeadTAsIsShownOnlyToThem() {
		eventInfo.setStatus(RequestStatus.FORWARDED);
		eventInfo.setForwardingGroup(ForwardingGroup.TEACHER_OR_HEAD_TA);

		Assertions.assertTrue(rs.shouldShowRequest(request, teacher));
		Assertions.assertFalse(rs.shouldShowRequest(request, ta));
	}

	@Test
	void testForwardedToPersonIsShownOnlyToThem() {
		eventInfo.setStatus(RequestStatus.FORWARDED);
		eventInfo.setForwardingGroup(ForwardingGroup.PERSON);
		eventInfo.setForwardedTo(taId);

		Assertions.assertTrue(rs.shouldShowRequest(request, ta));
		Assertions.assertFalse(rs.shouldShowRequest(request, teacher));
	}

	@Test
	void testFilterAssignmentsForWhichIsTeacherOrHeadTA() {
		Long editionId2 = 1001L;
		Long assignmentId2 = 2002L;
		Long moduleId2 = 3003L;

		EditionSummaryDTO edition2 = new EditionSummaryDTO().id(editionId2);
		EditionDetailsDTO editionDetails2 = new EditionDetailsDTO().id(editionId2);

		ModuleSummaryDTO module2 = new ModuleSummaryDTO().id(moduleId2);
		ModuleDetailsDTO moduleDetails2 = new ModuleDetailsDTO().id(moduleId2).edition(edition2);

		AssignmentDetailsDTO assignment2 = new AssignmentDetailsDTO().id(assignmentId2).module(module2);
		AssignmentSummaryDTO assignmentSummary = new AssignmentSummaryDTO().id(assignment.getId());
		AssignmentSummaryDTO assignmentSummary2 = new AssignmentSummaryDTO().id(assignmentId2);
		List<AssignmentSummaryDTO> assignments = List.of(assignmentSummary, assignmentSummary2);

		when(aCache.getRequired(assignmentId2)).thenReturn(assignment2);
		when(mCache.getRequired(moduleId2)).thenReturn(moduleDetails2);
		when(eCache.getRequired(editionId2)).thenReturn(editionDetails2);

		Assertions.assertEquals(as.assignmentsForWhichIsTeacherHeadTA(assignments, teacherId),
				List.of(assignment.getId()));
	}
}
